import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod} from 'bla';

import IApi from '../../common/interfaces/api/IApi';
import IApiBugReportRequest from '../../common/interfaces/api/IApiBugReportRequest';
import HttpMethod from '../../common/interfaces/HttpMethod';

import {apiTimedGot} from '../helpers/timedGot';
import requestIsFailed from './utils/requestIsFailed';
import logger from '../logger';
import getStaffLogin from '../helpers/getStaffLogin';
import checkIsYandexNetwork from '../helpers/checkIsYandexNetwork';
import isExpressRequest from '../../common/lib/isExpressRequest';

const config = yaConfig();
const startrekConfig = config.startrek;

const wrapJson = (str: string): string => `%%(json)${str}%%`;

const formatDescription = (
    description: string,
    meta: IApiBugReportRequest['meta'],
): string => {
    return [description, wrapJson(JSON.stringify(meta, null, 2))].join('\n');
};

const wrapProblem = (problem: string): string => `[${problem}]`;

const formatSummary = (problem: string, title: string): string => {
    return `${wrapProblem(problem)} ${title}`;
};

const action: IApi['execBugReport'] = async (
    {title, problem, description, meta},
    req,
) => {
    try {
        const authorLogin =
            req && isExpressRequest(req) ? getStaffLogin(req) : undefined; // логин на стаффе, есле привязан
        const isYandexNetwork =
            req && isExpressRequest(req) ? checkIsYandexNetwork(req) : false;

        const allowSendReport = Boolean(authorLogin) || isYandexNetwork;

        if (!allowSendReport) {
            const err = new Error('User not allowed to send report');

            logger.error('server/api/startrekApi', err);

            throw err;
        }

        if (!startrekConfig.key) {
            const err = new Error('No startrek api key provided');

            logger.error('server/api/startrekApi', err);

            throw err;
        }

        const requiredData = {
            title,
            problem,
            description,
        };

        if (
            !Object.keys(requiredData).every(key => Boolean(requiredData[key]))
        ) {
            const err = new Error('No required data provided');

            logger.error('server/api/startrekApi', err, {
                message: JSON.stringify(requiredData),
            });

            throw err;
        }

        const {body} = await apiTimedGot(
            {
                ...startrekConfig.options,
                method: HttpMethod.post,
                path: '/v2/issues/',
                json: true,
                body: {
                    queue: startrekConfig.queue,
                    type: 1, // bug
                    components: startrekConfig.components,
                    followers: authorLogin,
                    summary: formatSummary(
                        requiredData.problem,
                        requiredData.title,
                    ),
                    description: formatDescription(
                        requiredData.description,
                        meta,
                    ),
                },
                headers: {
                    Authorization: `OAuth ${startrekConfig.key}`,
                },
            },
            req,
            {
                operationName: '/v2/issues/',
            },
        );

        if (requestIsFailed(body)) {
            throw new Error(
                `Request is failed: ${JSON.stringify(body.errors)}`,
            );
        }

        return {success: true};
    } catch (err) {
        logger.error('server/api/bugReport', err, {level: 'error'});

        return {
            success: false,
        };
    }
};

module.exports = new ApiMethod({
    name: 'bugReport',
    params: {
        title: {
            type: 'String',
            required: true,
        },
        problem: {
            type: 'String',
            required: true,
        },
        description: {
            type: 'String',
            required: true,
        },
        meta: {
            type: 'Object',
            required: false,
        },
    },
    action,
});
