import {ApiMethod, ApiError} from 'bla';
import yaConfig from '@yandex-int/yandex-config';
import {Request} from 'express';

import EnvironmentType from '../../common/interfaces/EnvironmentType';

import IApiBusTariffs from '../../common/interfaces/api/IApiBusTariffs';
import ISearchBusTariffs from '../../common/interfaces/state/search/ISearchBusTariffs';
import IApiBusTariffsParams from '../../common/interfaces/api/IApiBusTariffsParams';
import IExpressRequest from '../../common/interfaces/IExpressRequest';

import requestIsFailed from './utils/requestIsFailed';
import logger from '../logger';
import isDirtyBot from '../helpers/isDirtyBot';
import {apiTimedGot} from '../helpers/timedGot';
import patchBusTariffs from '../helpers/busTariffs/patchBusTariffs';

const config = yaConfig();

const defaultAnswer: IApiBusTariffs = {
    busTariffs: {
        segments: [],
        querying: false,
    },
};

function getTimeout(environmentType: string | undefined): number {
    switch (true) {
        case typeof environmentType === 'undefined':
        case environmentType === EnvironmentType.client:
            return 15000;
        case environmentType === EnvironmentType.server:
            return 1000;

        default:
            return config.api.options.timeout;
    }
}

const action = async (
    {
        context,
        isCitySearch,
        nationalVersion,
        environmentType,
    }: IApiBusTariffsParams,
    req: Request & Pick<IExpressRequest, 'isBot' | 'isSearchBot'>,
): Promise<IApiBusTariffs> => {
    try {
        if (isDirtyBot(req)) {
            return {
                ...defaultAnswer,
                isDirty: true,
            };
        }

        const {from, to, when, language} = context;

        const response: {body: ISearchBusTariffs} = await apiTimedGot(
            {
                ...config.api.options,
                timeout: getTimeout(environmentType),
                path: `/${language}/segments/bus-tariffs/`,
                query: {
                    pointFrom: from.key,
                    pointTo: to.key,
                    date: [when.date, when.nextDate],
                    national_version: nationalVersion,
                    busSettlementKeys: isCitySearch,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/segments/bus-tariffs/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        if (!responseBody) {
            return defaultAnswer;
        }

        return {
            busTariffs: {
                ...responseBody,
                segments: patchBusTariffs(responseBody.segments || [], context),
            },
        };
    } catch (err) {
        if (
            err.code === 'ETIMEDOUT' &&
            environmentType === EnvironmentType.server
        ) {
            return {
                busTariffs: {
                    ...defaultAnswer.busTariffs,
                    busRequestTimeoutError: true,
                },
            };
        }

        logger.error('server/api/busTariffs', err);

        return defaultAnswer;
    }
};

module.exports = new ApiMethod({
    name: 'busTariffs2',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        environmentType: {
            type: 'String',
        },
        isCitySearch: {
            type: 'Boolean',
            required: false,
        },
        flags: {
            type: 'Object',
            required: false,
            defaultValue: {},
        },
    },
    action,
});
