import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';

import yaConfig from '@yandex-int/yandex-config';

import IApi from '../../common/interfaces/api/IApi';
import {FilterTransportType} from '../../common/lib/transportType';
import IApiCanonicalsResponse from '../../common/interfaces/api/IApiCanonicalsResponse';

import requestIsFailed from './utils/requestIsFailed';
import {apiTimedGot} from '../helpers/timedGot';

const config = yaConfig();

interface ICanonicalsResponse {
    body: {
        result: IApiCanonicalsResponse;
        errors: string | string[] | Record<string, string>;
    };
}

const {
    api: {options},
} = config;

const action: IApi['execCanonicals'] = async (
    {pointFrom, pointTo, transportType, language},
    req,
): Promise<IApiCanonicalsResponse | null> => {
    try {
        const response: ICanonicalsResponse = await apiTimedGot(
            {
                ...options,
                path: `/${language}/search/canonicals/`,
                query: {
                    pointFrom,
                    pointTo,
                    transportType:
                        transportType !== FilterTransportType.all
                            ? transportType
                            : undefined,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/search/canonicals/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result: resultFromResponse} = responseBody;

        if (!resultFromResponse) {
            return null;
        }

        if (!resultFromResponse.canonicals) {
            return null;
        }

        return resultFromResponse;
    } catch (err) {
        logger.error('server/api/canonicals.ts', err);

        return null;
    }
};

module.exports = new ApiMethod({
    name: 'canonicals',
    params: {
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        transportType: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
    },
    action,
});
