import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod} from 'bla';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';
import buildCurrencyRates from '../../common/lib/currencies/buildCurrencyRates';

const {api: apiConfig} = yaConfig();

module.exports = new ApiMethod({
    name: 'currencies',
    params: {
        language: {
            required: true,
            type: 'String',
        },
        nationalVersion: {
            required: true,
            type: 'String',
        },
        baseCurrency: {
            required: true,
            type: 'String',
        },
    },
    action: ({language, nationalVersion, baseCurrency}, req) => {
        const query = {
            national_version: nationalVersion,
            base: baseCurrency,
        };

        const defaultAnswer = {
            availableCurrencies: [baseCurrency],
            currencyRates: buildCurrencyRates([], baseCurrency),
        };

        return apiTimedGot(
            {
                ...apiConfig.options,
                path: `/${language}/currencies/`,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/currencies/',
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                const currencies = body.currencies.filter(
                    ({code, rate}) => rate && code !== baseCurrency,
                );

                // Базовая валюта - в начале списка
                const baseCurrencyObject = body.currencies.find(
                    ({code, rate}) => rate && code === baseCurrency,
                );

                if (baseCurrencyObject) {
                    currencies.unshift(baseCurrencyObject);
                }

                return {
                    availableCurrencies: currencies,
                    currencyRates: buildCurrencyRates(currencies, baseCurrency),
                };
            })
            .catch(err => {
                logger.error('server/api/currencies', err);

                return defaultAnswer;
            });
    },
});
