import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod} from 'bla';

import {apiTimedGot} from '../helpers/timedGot';
import isDirtyBot from '../helpers/isDirtyBot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';
import isAllDaysSearch from '../../common/lib/search/isAllDaysSearch';

const {api: apiConfig} = yaConfig();

const defaultAnswer = {
    isDefault: true,
    planeAllDaysTariffs: {
        segments: [],
        querying: false,
    },
};

module.exports = new ApiMethod({
    name: 'dynamicTariffs',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        clientSettlementId: {
            type: 'Number',
            required: true,
        },
        poll: {
            type: 'Boolean',
            defaultValue: false,
        },
    },
    action: ({context, nationalVersion, clientSettlementId, poll}, req) => {
        if (!poll && isDirtyBot(req)) {
            return {
                ...defaultAnswer,
                isDirty: true,
            };
        }

        const {transportType, from, to, when, language} = context;

        const query = {
            pointFrom: from.key,
            pointTo: to.key,
            date: [when.date, when.nextDate],
            national_version: nationalVersion,
            clientSettlementId,
        };

        let path;
        let operationName;

        if (isAllDaysSearch(context)) {
            path = `/${language}/segments/min-tariffs/`;
            operationName = '/:language/segments/min-tariffs/';
        } else {
            path = `/${language}/segments/tariffs/${poll ? 'poll/' : ''}`;
            operationName = `/:language/segments/tariffs/${
                poll ? 'poll/' : ''
            }`;
            query.date = [when.date, when.nextDate];
        }

        if (transportType !== 'all') {
            query.transportType = transportType;
        }

        return apiTimedGot(
            {
                ...apiConfig.options,
                path,
                query,
                json: true,
            },
            req,
            {
                operationName,
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                return {planeAllDaysTariffs: body};
            })
            .catch(err => {
                logger.error('server/api/planeAllDaysTariffs', err);

                return defaultAnswer;
            });
    },
});
