import {ApiMethod, ApiError} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {
    ALL_TRANSPORT_TYPES,
    TransportType,
    FilterTransportType,
} from '../../common/lib/transportType';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';
import {logSearchResult} from '../helpers/logs';
import isAllDaysSearch from '../../common/lib/search/isAllDaysSearch';
import mergeSegments from '../../common/lib/segments/mergeSegments';
import requestIsFailed from './utils/requestIsFailed';
import {excludeSuburbanSegmentsOfWrongPlan} from '../../common/lib/segments/suburbanPlans';
import patchSegmentsFromBackend from '../../common/lib/segments/patchSegmentsFromBackend';
import {parseSpecial} from '../../common/lib/date/parseImpl';
import IApi from '../../common/interfaces/api/IApi';
import NationalVersion from 'common/interfaces/NationalVersion';
import IApiSearchResponse from 'common/interfaces/api/IApiSearchResponse';
import Point from 'common/interfaces/Point';
import DateRobot from 'common/interfaces/date/DateRobot';
import ISegmentFromBackend from 'common/interfaces/segment/ISegmentFromBackend';
import Slug from 'common/interfaces/Slug';
import DateMoment from 'common/interfaces/date/DateMoment';
import DateSpecialValue from 'common/interfaces/date/DateSpecialValue';
import {TBannerInfo} from 'common/interfaces/state/search/BannerInfo';

const {api: apiConfig} = yaConfig();

interface ISearchRequestParams {
    /** @example 'c213' */
    pointFrom: Point;
    /** @example 'c2' */
    pointTo: Point;
    nationalVersion: NationalVersion;
    nearest: boolean;
    group_trains?: boolean;
    isMobile?: boolean;
    allowChangeContext?: boolean;
    /** Пока бэкенд возвращает отмены только с этим параметром для обратной совместимости */
    disable_cancels: false;
    /** @example [ 'Europe/Moscow' ] */
    timezones?: string[];
    transportType?: FilterTransportType;
    when?: DateRobot | DateSpecialValue;
}

interface ISearchResponseDirection {
    titleWithType: string;
    shortTitle: string;
    title: string;
    titleGenitive: string;
    titleLocative: string;
    preposition: string;
    key: Point;
    popularTitle: string;
    titleAccusative: string;
    slug: Slug;
}

interface ISearchResponse {
    body: {
        result: {
            plans: {
                current: {
                    startDate: DateRobot;
                    code: Point;
                    endDate: DateRobot;
                    title: string;
                };
                next: any | null;
            };
            teasers: object;
            segments: ISegmentFromBackend[];
            context: {
                transportTypes: TransportType[];
                search: {
                    nearest: boolean;
                    pointTo: ISearchResponseDirection;
                    pointFrom: ISearchResponseDirection;
                };
                latestDatetime: DateMoment;
                isChanged: boolean;
                original: {
                    nearest: boolean;
                    pointTo: ISearchResponseDirection;
                    pointFrom: ISearchResponseDirection;
                };
            };
            archivalData: any | null;
            canonical: {
                transportType: any | null;
                pointTo: Slug;
                pointFrom: Slug;
            };

            bannerInfo?: TBannerInfo;
        };
        errors: string | string[] | Record<string, string>;
    };
}

const action: IApi['execSearch'] = async (
    {
        context,
        isMobile,
        groupTrains,
        excludeTrains,
        nationalVersion,
        allowChangeContext,
    },
    req,
): Promise<IApiSearchResponse> => {
    const {
        transportType,
        from,
        to,
        when,
        searchNext = false,
        language,
        plan,
    } = context;

    const query: ISearchRequestParams = {
        pointFrom: from.key,
        pointTo: to.key,
        nationalVersion,
        nearest: searchNext,
        group_trains: groupTrains,
        isMobile,
        allowChangeContext,
        disable_cancels:
            false /* Пока бэкенд возвращает отмены только с этим параметром для обратной совместимости */,
    };

    if (from.timezone) {
        query.timezones = [from.timezone];
    }

    if (transportType !== FilterTransportType.all) {
        query.transportType = transportType;
    }

    if (when.date) {
        query.when = when.date;

        if (when.special) {
            query.when = parseSpecial(
                when.text,
                context.time,
                language,
                language,
                true,
            ).date;
        }
    }

    const isAllDays = isAllDaysSearch(context);
    const mergeIsAvailable =
        isAllDays &&
        !groupTrains &&
        transportType !== FilterTransportType.suburban;

    try {
        const response: ISearchResponse = await apiTimedGot(
            {
                ...apiConfig.options,
                timeout: 20000,
                path: `/${language}/search/search/`,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/search/search/',
            },
        );

        const {body} = response;

        if (requestIsFailed(body)) {
            throw body.errors;
        }

        const {result} = body;

        let segments = patchSegmentsFromBackend(result.segments || []);

        logSearchResult({req, query, result});

        if (isAllDays) {
            segments = excludeSuburbanSegmentsOfWrongPlan(
                {
                    segments,
                    plans: result.plans,
                },
                plan,
            );
        } else if (
            excludeTrains &&
            (transportType === FilterTransportType.all ||
                transportType === FilterTransportType.train)
        ) {
            segments = segments.filter(
                ({transport}) => transport.code !== TransportType.train,
            );
        }

        // проверяем, что в информации о канонической ссылке приходит понятный transportType
        const {canonical} = result;

        if (
            canonical &&
            canonical.transportType &&
            !ALL_TRANSPORT_TYPES.includes(canonical.transportType)
        ) {
            logger.error(
                '/server/api/search',
                `Unknown transportType "${canonical.transportType}"`,
            );
        }

        if (result.archivalData?.segments?.length) {
            segments = result.archivalData.segments.map(segment => {
                return {...segment, isArchival: true, isGone: false};
            });
        }

        const archivalData = result.archivalData
            ? {
                  transportTypes: result.archivalData.transportTypes,
                  canonical: result.archivalData.canonical,
                  hasSegments: Boolean(result.archivalData?.segments?.length),
              }
            : undefined;

        return {
            search: {
                ...result,
                archivalData,
                segments: mergeIsAvailable ? mergeSegments(segments) : segments,
            },
        };
    } catch (error) {
        logger.error('server/api/search', error);
        throw new ApiError();
    }
};

module.exports = new ApiMethod({
    name: 'search',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        isMobile: {
            type: 'Boolean',
            defaultValue: false,
        },
        excludeTrains: {
            type: 'Boolean',
            defaultValue: false,
        },
        groupTrains: {
            type: 'Boolean',
            defaultValue: false,
        },
        allowChangeContext: {
            type: 'Boolean',
            defaultValue: true,
        },
        // TODO: удалить после выкатки распов 18 мая 2022
        flags: {
            type: 'Object',
            required: false,
            defaultValue: {},
        },
    },
    action,
});
