import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {ALL_TYPE} from '../../common/lib/transportType';

import TransferQueryStatus from '../../common/interfaces/transfer/TransferQueryStatus';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';
import {buildSegmentsFromTransfers} from '../../common/lib/segments/transfer/buildSegmentsFromTransfers';
import {logSearchTransfersResult} from '../helpers/logs';
import generateIdForTransfer from '../../common/lib/segments/transfer/generateIdForTransfer';
import generateIdForTransferSegment from '../../common/lib/segments/transfer/generateIdForTransferSegment';

const {api: apiConfig} = yaConfig();

const defaultAnswer = {
    transfers: [],
    status: TransferQueryStatus.done,
};

module.exports = new ApiMethod({
    name: 'searchTransfer2',
    params: {
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
        when: {
            type: 'String',
            required: true,
        },
        transportType: {
            type: 'String',
            defaultValue: ALL_TYPE,
        },
    },
    action: ({pointFrom, pointTo, language, when, transportType}, req) => {
        const query = {
            pointFrom,
            pointTo,
            when,
            ...(transportType !== ALL_TYPE ? {transportType} : {}),
        };

        return apiTimedGot(
            {
                ...apiConfig.options,
                path: `/${language}/search/transfers/`,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/search/transfers/',
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                const {transferVariants} = body;

                logSearchTransfersResult({
                    req,
                    query,
                    result: transferVariants,
                });

                // добавляем id пересадкам, аналогично ручке transferWithPrices
                // это костыль и надеюсь, что временный
                const transfersWithId = transferVariants.map(transfer => ({
                    ...transfer,
                    id: generateIdForTransfer(transfer),
                    segments: transfer.segments.map(segment => ({
                        ...segment,
                        id: generateIdForTransferSegment(segment),
                    })),
                }));

                const transfers = buildSegmentsFromTransfers({
                    transfers: transfersWithId,
                    language,
                    transportType,
                });

                return {transfers, status: 'done'};
            })
            .catch(err => {
                logger.error('server/api/searchTransfer2', err);

                return defaultAnswer;
            });
    },
});
