import {ApiMethod} from 'bla';
import {URL} from 'url';

import yaConfig from '@yandex-int/yandex-config';
import environment from '@yandex-int/yandex-environment';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';

import {CacheByParamsInMap} from '../../common/lib/cache/CacheByParamsInMap';
import IApiSeoMetaInfoParams from '../../common/interfaces/api/IApiSeoMetaInfoParams';
import ApiSeoMetaInfoResponse from '../../common/interfaces/api/ApiSeoMetaInfoResponse';
import Req from '../../common/interfaces/router/Req';
import Environment from '../../common/interfaces/Environment';

const config = yaConfig();

const cacheConfig = {
    size: 1,
    lifetime: 5 * 60 * 1000,
};
const serverCache = new CacheByParamsInMap(cacheConfig);

const defaultAnswer = null;
const {bunker: bunkerConfig} = config;

module.exports = new ApiMethod({
    name: 'seoMetaInfo',
    params: {
        url: {
            type: 'String',
            required: true,
        },
    },
    action: (
        {url}: IApiSeoMetaInfoParams,
        req: Req,
    ): ApiSeoMetaInfoResponse => {
        const query = {
            node: `rasp/${
                environment === Environment.production
                    ? 'production'
                    : 'testing'
            }`,
        };

        const currentUrl = url || req.url;

        const metaInfoFromCache = serverCache.get({});

        if (metaInfoFromCache) {
            return metaInfoFromCache[currentUrl] ?? defaultAnswer;
        }

        return apiTimedGot(
            {
                ...bunkerConfig,
                path: '/v1/cat',
                query,
                json: true,
            },
            req,
            {
                operationName: '/v1/cat',
            },
        )
            .then(data => {
                const metaInfoList = data.body.schemas;

                if (!metaInfoList) {
                    serverCache.set({}, {});

                    return null;
                }

                let coincidingMetaInfoObject = null;
                const modifiedMetaInfoObject = {};

                metaInfoList.forEach(metaInfo => {
                    const metaInfoUrl = new URL(metaInfo.href);
                    const path = metaInfoUrl.pathname + metaInfoUrl.search;

                    modifiedMetaInfoObject[path] = {
                        ...metaInfo,
                        href: path,
                    };

                    if (currentUrl === path) {
                        coincidingMetaInfoObject = {
                            ...metaInfo,
                            href: path,
                        };
                    }
                });

                serverCache.set({}, modifiedMetaInfoObject);

                return coincidingMetaInfoObject;
            })
            .catch(err => {
                const status = err.statusCode || 500;

                logger.error('server/api/bunker', err, {
                    message: `Failed to send ${status}`,
                    tags: {http: 'important_bunker_error'},
                    level: 'error',
                });

                return defaultAnswer;
            });
    },
});
