import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod} from 'bla';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';
import isAllDaysSearch from '../../common/lib/search/isAllDaysSearch';

const {api: apiConfig} = yaConfig();

module.exports = new ApiMethod({
    name: 'staticTariffs',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        clientSettlementId: {
            type: 'Number',
            required: true,
        },
    },
    action: ({context, nationalVersion, clientSettlementId}, req) => {
        const {transportType, from, to, when, language} = context;

        const defaultAnswer = {
            staticTariffs: {
                tariffs: [],
            },
        };

        const query = {
            pointFrom: from.key,
            pointTo: to.key,
            national_version: nationalVersion,
            clientSettlementId,
        };

        let path;
        let operationName;

        if (isAllDaysSearch(context)) {
            path = `/${language}/tariffs/min-static-tariffs/`;
            operationName = '/:language/tariffs/min-static-tariffs/';
        } else {
            path = `/${language}/tariffs/static-tariffs/`;
            operationName = '/:language/tariffs/static-tariffs/';
            query.date = [when.date, when.nextDate];
        }

        if (transportType !== 'all') {
            query.transportType = transportType;
        }

        return apiTimedGot(
            {
                ...apiConfig.options,
                path,
                query,
                json: true,
            },
            req,
            {
                operationName,
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                return {staticTariffs: body};
            })
            .catch(err => {
                logger.error('server/api/staticTariffs', err);

                return defaultAnswer;
            });
    },
});
