import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';

const {api: apiConfig} = yaConfig();

const defaultAnswer = {
    staticTariffs: {
        tariffs: [],
    },
};

module.exports = new ApiMethod({
    name: 'staticTariffs2',
    params: {
        language: {
            type: 'String',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        startDate: {
            type: 'String', // format: YYYY-MM-DD
        },
        endDate: {
            type: 'String', // format: YYYY-MM-DD
        },
        transportType: {
            type: 'String',
        },
    },
    action: (
        {
            language,
            nationalVersion,
            pointFrom,
            pointTo,
            startDate,
            endDate,
            transportType = 'all',
        },
        req,
    ) => {
        const query = {
            pointFrom,
            pointTo,
            national_version: nationalVersion,
        };

        let path;
        let operationName;

        if (!startDate || !endDate) {
            path = `/${language}/tariffs/min-static-tariffs/`;
            operationName = '/:language/tariffs/min-static-tariffs/';
        } else {
            path = `/${language}/tariffs/static-tariffs/`;
            operationName = '/:language/tariffs/static-tariffs/';
            query.date = [startDate, endDate];
        }

        if (transportType !== 'all') {
            query.transportType = transportType;
        }

        return apiTimedGot(
            {
                ...apiConfig.options,
                path,
                query,
                json: true,
            },
            req,
            {
                operationName,
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                return {staticTariffs: body};
            })
            .catch(err => {
                logger.error('server/api/staticTariffs2', err);

                return defaultAnswer;
            });
    },
});
