import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';
import moment from 'moment';

import yaConfig from '@yandex-int/yandex-config';

import IStationFromApi from '../../common/interfaces/state/station/IStationFromApi';
import IStationFromBackend from '../../common/interfaces/state/station/IStationFromBackend';
import StationSubtype from '../../common/interfaces/state/station/StationSubtype';
import StationEventList from '../../common/interfaces/state/station/StationEventList';
import IStationWayToAirport from '../../common/interfaces/state/station/IStationWayToAirport';
import StationDateSpecialValue from '../../common/interfaces/date/StationDateSpecialValue';
import StopsType from '../../common/interfaces/state/station/StopsType';
import {TransportType} from '../../common/lib/transportType';
import TeaserType from '../../common/interfaces/state/teasers/TeaserType';
import ITeaserFromApi from '../../common/interfaces/state/teasers/ITeaserFromApi';
import IApi from '../../common/interfaces/api/IApi';
import TransportSubtypeCode from '../../common/interfaces/state/station/TransportSubtypeCode';
import ThreadStatus from '../../common/interfaces/state/station/ThreadStatus';
import StationType from '../../common/interfaces/state/station/StationType';

import getTypeStation from './utils/getTypeStation';
import getValueFromEnum from '../../common/lib/enum/getValueFromEnum';
import getArrayValuesFromEnum from '../../common/lib/enum/getArrayValuesFromEnum';
import isDateRobot from '../../common/lib/date/isDateRobot';
import isPoint from '../../common/lib/point/isPoint';
import isDateMoment from '../../common/lib/date/isDateMoment';
import requestIsFailed from './utils/requestIsFailed';
import isSlug from '../../common/lib/isSlug';

import {apiTimedGot} from '../helpers/timedGot';

interface IStationResponse {
    body: {
        result: IStationFromBackend;
        errors: string | string[] | Record<string, string>;
    };
}

const config = yaConfig();

const action: IApi['execStation'] = async (
    {
        stationId,
        language,
        isMobile,
        subtype,
        date: dateFromParam,
        event: eventFromParam,
        direction: directionFromParam,
        nationalVersion,
    },
    req,
): Promise<IStationFromApi> => {
    try {
        const {
            api: {options},
        } = config;

        const response: IStationResponse = await apiTimedGot(
            {
                ...options,
                timeout: 15000,
                path: `/${language}/station/`,
                query: {
                    station_id: stationId,
                    subtype,
                    is_mobile: isMobile,
                    event: eventFromParam,
                    date: dateFromParam,
                    direction: directionFromParam,
                    country: nationalVersion,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/station/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result: resultFromBackend} = responseBody;

        const {
            station: {
                id,
                title,
                titleGenitive,
                fullTitle,
                fullTitleDative,
                fullTitleGenitive,
                hasPopularTitle,
                subway,
                address,
                settlement,
                longitude,
                latitude,
                wayToAirport: wayToAirportFromBackend,
                directions,
                trusted,
                iataCode,
                stationType: pointType,
                phones,
            },
            pageType: {
                type: typeFromBackend,
                subtypes: subtypesFromBackend,
                mainSubtype: mainSubtypeFromBackend,
                currentSubtype: currentSubtypeFromBackend,
                terminals = [],
                notEnoughInfo,
            },
            context: {event: eventFromBackend, directionCode, when, dtNow},
            threads = [],
            teasers,
            companies: companiesFromBackend = [],
            scheduleBlocks = [],
        } = resultFromBackend;

        const type = getTypeStation(typeFromBackend);

        if (!type) {
            throw new Error(
                `Неизвестный тип станции. pageType.type: ${typeFromBackend}`,
            );
        }

        // Это вспомогательная переменная чтобы понимать что с бекенда вернулась
        // самолетная станция в которой есть хотя бы одна самолетная нитка.
        // Нужно это для того, чтобы корректно заполнить subtype-поля, потому что с
        // бекенда может придти самолетная страница с самолетными нитками, но при этом данные поля заполнены
        // не будут. Так что мы пока чутка закостылим это на фронте
        const isPlanePageWithPlaneThreads =
            type === StationType.plane &&
            threads.length &&
            threads[0].transportType === TransportType.plane;

        const subtypes = getArrayValuesFromEnum(
            subtypesFromBackend,
            StationSubtype,
        );

        if (subtypes.length !== subtypesFromBackend.length) {
            throw new Error(
                `Неизвестный подтип станции. pageType.subtype: ${subtypesFromBackend}`,
            );
        }

        if (subtypes.length === 0 && isPlanePageWithPlaneThreads) {
            subtypes.push(StationSubtype.plane);
        }

        const mainSubtype =
            mainSubtypeFromBackend !== null
                ? getValueFromEnum(mainSubtypeFromBackend, StationSubtype)
                : isPlanePageWithPlaneThreads
                ? StationSubtype.plane
                : undefined;

        const currentSubtype =
            currentSubtypeFromBackend !== null
                ? getValueFromEnum(currentSubtypeFromBackend, StationSubtype)
                : isPlanePageWithPlaneThreads
                ? StationSubtype.plane
                : undefined;

        const event = getValueFromEnum(eventFromBackend, StationEventList);

        if (!event) {
            throw new Error(
                `Не известный event станции. event: ${eventFromBackend}`,
            );
        }

        const whenDate = when.date;

        if (whenDate !== undefined && !isDateRobot(whenDate)) {
            throw new Error(
                `Формат даты не соответствует ожидаемому. when.date: ${when.date}`,
            );
        }

        const dtAfter = when.dtAfter;

        if (dtAfter !== undefined && !isDateMoment(dtAfter)) {
            throw new Error(
                `Формат даты не соответствует ожидаемому. when.dtAfter: ${when.dtAfter}`,
            );
        }

        const dtBefore = when.dtBefore;

        if (dtBefore !== undefined && !isDateMoment(dtBefore)) {
            throw new Error(
                `Формат даты не соответствует ожидаемому. when.dtBefore: ${when.dtBefore}`,
            );
        }

        const now = dtNow;

        if (!isDateMoment(now)) {
            throw new Error(
                `Формат даты не соответствует ожидаемому. context.dtNow: ${now}`,
            );
        }

        const whenSpecial =
            when.special !== undefined
                ? getValueFromEnum(when.special, StationDateSpecialValue)
                : undefined;

        const wayToAirport = getWayToAirport(wayToAirportFromBackend);
        const teasersAsArray = Object.entries(teasers).reduce<ITeaserFromApi[]>(
            (result, [key, value], index) => {
                const teaserType = getValueFromEnum(key, TeaserType);

                if (teaserType) {
                    result.push({
                        ...value,
                        id: value.id ?? index,
                        type: teaserType,
                        mobileContent: value.mobileContent ?? undefined,
                    });
                }

                return result;
            },
            [],
        );

        const momentNow = moment(now);

        const threadsFromApi = threads.map(thread => {
            const {
                eventDt: {datetime: eventDtDate},
                number,
                title: threadTitle,
                transportType,
                canonicalUid,
                isAeroExpress,
                isExpress,
                stops: threadStops,
                departure,
                departureFrom,
                transportSubtype,
                deluxeTrainTitle,
                platform,
                daysText,
                companyId,
                terminal,
                aviaLink,
                routeStations,
                codeshares,
                isSupplement,
                status,
            } = thread;
            const threadStopsType = threadStops?.type;

            if (eventDtDate !== undefined && !isDateMoment(eventDtDate)) {
                throw new Error(
                    `Формат даты не соответствует ожидаемому. threadUid: ${canonicalUid}. eventDt.datetime: ${thread.eventDt.datetime}`,
                );
            }

            if (departure !== undefined && !isDateRobot(departure)) {
                throw new Error(
                    `Формат даты не соответствует ожидаемому. threadUid: ${canonicalUid}. departure: ${thread.departure}`,
                );
            }

            if (departureFrom !== undefined && !isDateMoment(departureFrom)) {
                throw new Error(
                    `Формат даты не соответствует ожидаемому. threadUid: ${canonicalUid}. departureFrom: ${thread.departureFrom}`,
                );
            }

            const transportTypeFromApi = getValueFromEnum(
                transportType,
                TransportType,
            );

            if (!transportTypeFromApi) {
                throw new Error(
                    `Формат типа транспорта не соответствует ожидаемому. threadUid: ${canonicalUid}. transportType: ${transportType}`,
                );
            }

            let threadStopsTypeFromApi;

            if (threadStopsType) {
                threadStopsTypeFromApi = getValueFromEnum(
                    threadStopsType,
                    StopsType,
                );

                if (!threadStopsTypeFromApi) {
                    throw new Error(
                        `Формат типа остановок не соответствует ожидаемому. threadUid: ${canonicalUid}. stops.type: ${threadStopsType}`,
                    );
                }
            }

            const statusType =
                status && getValueFromEnum(status.status, ThreadStatus);

            if (status && !statusType) {
                throw new Error(
                    `Формат статуса нитки не соответствует ожидаемому. threadUid: ${canonicalUid}. status: ${JSON.stringify(
                        status,
                    )}`,
                );
            }

            const momentEventDt = eventDtDate && moment(eventDtDate);

            return {
                eventDt: {
                    ...thread.eventDt,
                    datetime: eventDtDate,
                },
                number,
                title: threadTitle,
                transportType: transportTypeFromApi,
                canonicalUid,

                isExpress,
                isAeroExpress,
                stops: threadStops
                    ? {
                          text: threadStops.text,
                          type: threadStopsTypeFromApi,
                      }
                    : undefined,
                departure,
                departureFrom,
                transportSubtype: transportSubtype
                    ? {
                          ...transportSubtype,
                          code: getValueFromEnum(
                              transportSubtype?.code,
                              TransportSubtypeCode,
                          ),
                      }
                    : undefined,
                deluxeTrainTitle,
                platform,
                daysText,
                companyId,
                terminalName: terminal || undefined,
                aviaLink,
                routeStations,
                codeshares,
                isSupplement,
                status: status
                    ? {
                          status: statusType ?? ThreadStatus.unknown,
                          actualDt: status.actualDt || undefined,
                          actualTerminalName:
                              status.actualTerminal || undefined,
                          checkInDesks: status.checkInDesks || undefined,
                          gate: status.gate || undefined,
                          baggageCarousels:
                              status.baggageCarousels || undefined,
                          diverted: status.diverted || undefined,
                      }
                    : undefined,
                hoursBeforeEvent:
                    type === StationType.plane && momentEventDt
                        ? momentEventDt.diff(momentNow, 'hours', true)
                        : undefined,
                minutesBetweenEventDtAndActualDt:
                    type === StationType.plane &&
                    status?.actualDt &&
                    momentEventDt
                        ? momentEventDt.diff(
                              moment(status.actualDt),
                              'minutes',
                              true,
                          )
                        : undefined,
            };
        });

        const companies = companiesFromBackend.map(company => ({
            ...company,
            shortTitle: company.shortTitle || undefined,
            icon: company.icon || undefined,
        }));

        return {
            id,
            type,
            subtypes,
            pointType,
            title,
            hasPopularTitle,
            longitude: longitude ?? undefined,
            latitude: latitude ?? undefined,
            directions,
            directionCode,
            event,
            whenDate,
            threads: threadsFromApi,
            teasers: teasersAsArray,
            trusted: Boolean(trusted),
            companies,
            terminals,
            now,
            scheduleBlocks,

            mainSubtype,
            currentSubtype,
            subway: subway ?? undefined,
            address: address ?? undefined,
            settlement: settlement
                ? {
                      ...settlement,
                      titleGenitive: settlement.titleGenitive ?? undefined,
                  }
                : undefined,
            titleGenitive: titleGenitive || undefined,
            fullTitle: fullTitle || undefined,
            fullTitleDative: fullTitleDative || undefined,
            fullTitleGenitive: fullTitleGenitive || undefined,
            whenSpecial,
            wayToAirport,
            dtAfter,
            dtBefore,
            iataCode,
            phones,
            notEnoughInfo,
        };
    } catch (err) {
        logger.error('server/api/station.js', err);

        throw err;
    }
};

module.exports = new ApiMethod({
    name: 'station',
    params: {
        stationId: {
            type: 'Number',
            required: true,
        },
        isMobile: {
            type: 'Boolean',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },

        date: {
            type: 'String',
        },
        event: {
            type: 'String',
        },
        subtype: {
            type: 'String',
        },
        direction: {
            type: 'String',
        },
        nationalVersion: {
            type: 'String',
        },
    },
    action,
});

function getWayToAirport(
    wayToAirport: IStationFromBackend['station']['wayToAirport'],
): IStationWayToAirport | undefined {
    if (!wayToAirport) {
        return;
    }

    const {toPointId, fromPointId, fromSlug, toSlug, linkTitle, wayType} =
        wayToAirport;

    if (!isPoint(toPointId) || !isPoint(fromPointId)) {
        throw new Error(
            `Пункты прибытия/назначения не соответствует требуемому формату. wayToAirport: ${JSON.stringify(
                wayToAirport,
            )}`,
        );
    }

    if (!isSlug(fromSlug) || !isSlug(toSlug)) {
        throw new Error(
            `Слаги пунктов прибытия/назначения не соответствует требуемому формату. wayToAirport: ${JSON.stringify(
                wayToAirport,
            )}`,
        );
    }

    return {
        fromPoint: fromPointId,
        toPoint: toPointId,
        fromSlug,
        toSlug,
        title: linkTitle,
        wayType: wayType ?? undefined,
    };
}
