import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';

import yaConfig from '@yandex-int/yandex-config';

import IStationCityStationsFromBackend from '../../common/interfaces/state/station/IStationCityStationsFromBackend';
import ICityData from '../../common/interfaces/state/station/ICityData';
import IApi from '../../common/interfaces/api/IApi';

import requestIsFailed from './utils/requestIsFailed';
import {apiTimedGot} from '../helpers/timedGot';

interface IStationCityStationResponse {
    body: {
        result: IStationCityStationsFromBackend;
        errors: string | string[] | Record<string, string>;
    };
}

const config = yaConfig();

const action: IApi['execStationCityStations'] = async (
    {stationId, language},
    req,
): Promise<ICityData | undefined> => {
    try {
        const {
            api: {options},
        } = config;

        const response: IStationCityStationResponse = await apiTimedGot(
            {
                ...options,
                path: `/${language}/station/city_stations/`,
                query: {
                    station_id: stationId,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/station/city_stations/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result} = responseBody;

        if (!result) {
            return undefined;
        }

        return {
            settlement:
                result.settlement === null ? undefined : result.settlement,
            cityStations: result.cityStations,
        };
    } catch (err) {
        logger.error('server/api/stationCityStations.js', err);

        return undefined;
    }
};

module.exports = new ApiMethod({
    name: 'stationCityStations',
    params: {
        stationId: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
    },
    action,
});
