import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';

import yaConfig from '@yandex-int/yandex-config';

import IStationPopularDirectionsFromApi from '../../common/interfaces/state/station/IStationPopularDirectionsFromApi';
import IStationPopularDirectionsFromBackend from '../../common/interfaces/state/station/IStationPopularDirectionsFromBackend';
import IPopularDirectionStation from '../../common/interfaces/state/station/IPopularDirectionStation';
import IPopularDirectionStationFromBackend from '../../common/interfaces/state/station/IPopularDirectionStationFromBackend';
import IApi from '../../common/interfaces/api/IApi';
import {FilterTransportType} from '../../common/lib/transportType';

import getValueFromEnum from '../../common/lib/enum/getValueFromEnum';
import isPoint from '../../common/lib/point/isPoint';
import requestIsFailed from './utils/requestIsFailed';
import {apiTimedGot} from '../helpers/timedGot';
import isSlug from '../../common/lib/isSlug';

interface IStationPopularDirectionsResponse {
    body: {
        result: IStationPopularDirectionsFromBackend;
        errors: string | string[] | Record<string, string>;
    };
}

const config = yaConfig();

const action: IApi['execStationPopularDirections'] = async (
    {subtype, language, limit, stationId},
    req,
): Promise<IStationPopularDirectionsFromApi | undefined> => {
    try {
        const {
            api: {options},
        } = config;

        const response: IStationPopularDirectionsResponse = await apiTimedGot(
            {
                ...options,
                path: `/${language}/station/popular_directions/`,
                query: {
                    station_id: stationId,
                    subtype,
                    limit,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/station/popular_directions/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result: resultFromResponse} = responseBody;

        if (!resultFromResponse) {
            return undefined;
        }

        const {searchTransportType, station, from, to} = resultFromResponse;
        const transportType = getValueFromEnum(
            searchTransportType,
            FilterTransportType,
        );
        const stationPoint = station.key;
        const stationSlug = station.slug;
        const formatStationArray = (
            array: IPopularDirectionStationFromBackend[],
        ): IPopularDirectionStation[] => {
            return array.reduce<IPopularDirectionStation[]>(
                (result, toStation) => {
                    const {key: toPoint, slug: toSlug} = toStation;

                    if (isPoint(toPoint) && isSlug(toSlug)) {
                        result.push({
                            ...toStation,
                            point: toPoint,
                            slug: toSlug,
                        });
                    }

                    return result;
                },
                [],
            );
        };
        const toTyped = formatStationArray(to);
        const fromTyped = formatStationArray(from);

        if (!transportType) {
            throw new Error(
                `Формат типа транспорта не соответствует ожидаемому. stationPopularDirections.searchTransportType: ${transportType}`,
            );
        }

        if (!isPoint(stationPoint)) {
            throw new Error(
                `Станция не соответствует требуемому формату. stationPopularDirections.station.key: ${JSON.stringify(
                    stationPoint,
                )}`,
            );
        }

        if (!isSlug(stationSlug)) {
            throw new Error(
                `Слаг станции не соответствует требуемому формату. stationPopularDirections.station.key: ${JSON.stringify(
                    stationPoint,
                )}`,
            );
        }

        return {
            searchTransportType: transportType,
            station: {
                ...station,
                point: stationPoint,
                slug: stationSlug,
            },
            to: toTyped,
            from: fromTyped,
        };
    } catch (err) {
        logger.error('server/api/stationPopularDirections.js', err);

        return undefined;
    }
};

module.exports = new ApiMethod({
    name: 'stationPopularDirections',
    params: {
        stationId: {
            type: 'String',
            required: true,
        },
        subtype: {
            type: 'String',
            required: true,
        },
        limit: {
            type: 'Number',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
    },
    action,
});
