import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';

import yaConfig from '@yandex-int/yandex-config';

import IStationQuickFromApi from '../../common/interfaces/state/station/IStationQuickFromApi';
import IStationQuickFromBackend from '../../common/interfaces/state/station/IStationQuickFromBackend';
import StationSubtype from '../../common/interfaces/state/station/StationSubtype';
import IApi from '../../common/interfaces/api/IApi';

import getArrayValuesFromEnum from '../../common/lib/enum/getArrayValuesFromEnum';
import getValueFromEnum from '../../common/lib/enum/getValueFromEnum';
import requestIsFailed from './utils/requestIsFailed';
import getTypeStation from './utils/getTypeStation';
import {apiTimedGot} from '../helpers/timedGot';

interface IStationQuickResponse {
    body: {
        result: IStationQuickFromBackend;
        errors: string | string[] | Record<string, string>;
    };
}

const config = yaConfig();

const action: IApi['execStationQuick'] = async (
    {stationId, language, subtype},
    req,
): Promise<IStationQuickFromApi> => {
    try {
        const {
            api: {options},
        } = config;

        const response: IStationQuickResponse = await apiTimedGot(
            {
                ...options,
                path: `/${language}/station/quick/`,
                query: {
                    station_id: stationId,
                    subtype,
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/station/quick/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result} = responseBody;

        const {
            type: typeFromBackend,
            subtypes: subtypesFromBackend,
            notEnoughInfo,
            mainSubtype: mainSubtypeFromBackend,
            currentSubtype: currentSubtypeFromBackend,
            terminals,
        } = result;

        const type = getTypeStation(typeFromBackend);

        if (!type) {
            throw new Error(
                `Неизвестный тип станции. type: ${typeFromBackend}`,
            );
        }

        const subtypes = getArrayValuesFromEnum(
            subtypesFromBackend,
            StationSubtype,
        );

        if (subtypes.length !== subtypesFromBackend.length) {
            throw new Error(
                `Неизвестный подтип станции. subtype: ${subtypesFromBackend}`,
            );
        }

        const mainSubtype =
            mainSubtypeFromBackend !== null
                ? getValueFromEnum(mainSubtypeFromBackend, StationSubtype)
                : undefined;

        const currentSubtype =
            currentSubtypeFromBackend !== null
                ? getValueFromEnum(currentSubtypeFromBackend, StationSubtype)
                : undefined;

        return {
            type,
            terminals,
            subtypes,
            notEnoughInfo,
            currentSubtype,
            mainSubtype,
        };
    } catch (err) {
        logger.error('server/api/stationQuick.js', err);

        throw err;
    }
};

module.exports = new ApiMethod({
    name: 'stationQuick',
    params: {
        stationId: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },

        subtype: {
            type: 'String',
        },
    },
    action,
});
