import {ApiMethod, ApiError} from 'bla';
import logger from '../logger';

import yaConfig from '@yandex-int/yandex-config';

import IApi from '../../common/interfaces/api/IApi';
import IStationStopsFromBackend from '../../common/interfaces/state/station/IStationStopsFromBackend';

import requestIsFailed from './utils/requestIsFailed';
import {apiTimedGot} from '../helpers/timedGot';

interface IStationResponse {
    body: {
        result: IStationStopsFromBackend;
        errors: string | string[] | Record<string, string>;
    };
}

const config = yaConfig();

const action: IApi['execStationStops'] = async (
    {
        stationId,
        language,
        subtype,
        date: dateFromParam,
        event: eventFromParam,
        direction: directionFromParam,
        nationalVersion,
        returnForTypes,
    },
    req,
): Promise<IStationStopsFromBackend> => {
    try {
        const {
            api: {options},
        } = config;

        const response: IStationResponse = await apiTimedGot(
            {
                ...options,
                timeout: 10000,
                path: `/${language}/station/stops/`,
                query: {
                    station_id: stationId,
                    subtype,
                    event: eventFromParam,
                    date: dateFromParam,
                    direction: directionFromParam,
                    country: nationalVersion,
                    return_for_types: returnForTypes?.join(','),
                },
                json: true,
            },
            req,
            {
                operationName: '/:language/station/stops/',
            },
        );

        const {body: responseBody} = response;

        if (requestIsFailed(responseBody)) {
            throw new ApiError(
                null,
                `Request failed: ${JSON.stringify(responseBody)}`,
            );
        }

        const {result: resultFromBackend} = responseBody;

        const {stops = []} = resultFromBackend;

        return {
            stops,
        };
    } catch (err) {
        logger.error('server/api/station.js', err);

        throw err;
    }
};

module.exports = new ApiMethod({
    name: 'stationStops',
    params: {
        stationId: {
            type: 'Number',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },

        date: {
            type: 'String',
        },
        event: {
            type: 'String',
        },
        subtype: {
            type: 'String',
        },
        direction: {
            type: 'String',
        },
        nationalVersion: {
            type: 'String',
        },
        returnForTypes: {
            type: 'Array',
        },
    },
    action,
});
