import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';

const {api: apiConfig} = yaConfig();

module.exports = new ApiMethod({
    name: 'threadMap',
    params: {
        threadId: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
        country: {
            type: 'String',
        },
        departure: {
            type: 'String',
        },
        departureFrom: {
            type: 'String',
        },
        stationFrom: {
            type: 'String',
        },
        stationTo: {
            type: 'String',
        },
        timezone: {
            type: 'String',
            defaultValue: '',
        },
    },
    action: (
        {
            threadId,
            country,
            language,
            departure,
            departureFrom,
            stationFrom,
            stationTo,
            timezone,
        },
        req,
    ) => {
        const query = {
            mixed_uid: threadId,
            country,
            departure,
            departure_from: departureFrom,
            station_from: stationFrom,
            station_to: stationTo,
            time_zone: timezone,
        };

        const path = `/${language}/thread/map/`;

        return apiTimedGot(
            {
                ...apiConfig.options,
                path,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/thread/map/',
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                return body.result;
            })
            .catch(err => {
                // Не логируем 404, потому что это часть ожидаемого функционала
                if (err.statusCode !== 404) {
                    logger.error('server/api/threadMap.js', err);
                }

                throw err;
            });
    },
});
