import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {SUBURBAN_TYPE} from '../../common/lib/transportType';

import {TrainPartner} from '../../common/lib/order/trainPartners';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';
import isDirtyBot from '../helpers/isDirtyBot';

import requestIsFailed from './utils/requestIsFailed';
import shouldRemoveTariffs from '../../common/lib/segments/shouldRemoveTariffs';

const {api: apiConfig} = yaConfig();

function patchStationTitle(station) {
    const {ufsTitle} = station || {};

    return ufsTitle ? {...station, popularTitle: ufsTitle} : station;
}

function patchCompanyTitle(company) {
    const {ufsTitle} = company || {};

    return ufsTitle ? {...company, shortTitle: ufsTitle} : company;
}

function patchSegmentTitle(segment) {
    return segment.ufsTitle || segment.title;
}

function patchSegmentTitles(segment) {
    segment.stationFrom = patchStationTitle(segment.stationFrom);
    segment.stationTo = patchStationTitle(segment.stationTo);
    segment.company = patchCompanyTitle(segment.company);
    segment.title = patchSegmentTitle(segment);

    return segment;
}

const defaultAnswer = {
    isDefault: true,
    trainTariffs: {
        segments: [],
        querying: false,
    },
};

module.exports = new ApiMethod({
    name: 'trainTariffs',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        startTime: {
            type: 'String',
        },
        endTime: {
            type: 'String',
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        poll: {
            type: 'Boolean',
            defaultValue: false,
        },
        /**
         * @todo Со временем небходимо сделать required: true.
         * Оставили для того, чтобы боты, которые обращаются к api не создавали фон ошибок.
         */
        buyOnRasp: {
            type: 'Boolean',
            required: false,
        },
        /**
         * @todo Со временем небходимо сделать required: true.
         * Оставили для того, чтобы боты, которые обращаются к api не создавали фон ошибок.
         */
        trainProviderTesting: {
            type: 'Boolean',
            required: false,
        },
        /**
         * @todo Со временем небходимо убрать параметр flags.
         * Оставили для того, чтобы боты, которые обращаются к api не создавали фон ошибок.
         */
        flags: {
            type: 'Object',
            required: false,
            defaultValue: {},
        },
    },
    action: (
        {
            buyOnRasp,
            trainProviderTesting,
            context,
            startTime,
            endTime,
            nationalVersion,
            poll,
            flags,
        },
        req,
    ) => {
        if (!poll && isDirtyBot(req)) {
            return {
                ...defaultAnswer,
                isDirty: true,
            };
        }

        const {to, from, when, time, language, transportType} = context;
        const path = `/${language}/api/segments/train-tariffs/${
            poll ? 'poll/' : ''
        }`;

        let query = {
            pointFrom: from.key,
            pointTo: to.key,
            // управление комиссией Яндекса
            experiment: Boolean(flags.__experiment),
            national_version: nationalVersion,
            includePriceFee: buyOnRasp || flags.__buyOnRasp ? '1' : '0',
            partner: TrainPartner.im,
        };

        if (trainProviderTesting || flags.__ufsTesting) {
            query = {
                ...query,
                queriesLimit: 1,
                date: when.date,
                useRailwayTZ: true,
            };
        } else {
            query = {
                ...query,
                startTime,
                endTime,
            };
        }

        return apiTimedGot(
            {
                ...apiConfig.purchaseOptions,
                path,
                query,
                json: true,
            },
            req,
            {
                operationName: `/:language/api/segments/train-tariffs/${
                    poll ? 'poll/' : ''
                }`,
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                if (body.segments) {
                    body.segments = body.segments
                        .filter(segment => !shouldRemoveTariffs(segment, time))
                        .map(segment => {
                            if (transportType === SUBURBAN_TYPE) {
                                segment.canSupplySegments = false;
                            }

                            return segment;
                        });

                    if (trainProviderTesting) {
                        body.segments = body.segments.map(patchSegmentTitles);
                    }
                }

                return {trainTariffs: body};
            })
            .catch(err => {
                logger.error('server/api/trainTariffs', err);

                return defaultAnswer;
            });
    },
});
