import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {SUBURBAN_TYPE} from '../../common/lib/transportType';

import {TrainPartner} from '../../common/lib/order/trainPartners';
import EnvironmentType from '../../common/interfaces/EnvironmentType';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';
import isDirtyBot from '../helpers/isDirtyBot';

import requestIsFailed from './utils/requestIsFailed';
import {getCommonHeaders} from './utils/getCommonHeaders';
import shouldRemoveTariffs from '../../common/lib/segments/shouldRemoveTariffs';
import patchSegmentsFromBackend from '../../common/lib/segments/patchSegmentsFromBackend';

const {api: apiConfig} = yaConfig();

const defaultAnswer = {
    isDefault: true,
    trainTariffs: {
        segments: [],
        querying: false,
    },
};

function getTimeout({environmentType, poll}) {
    switch (true) {
        case typeof environmentType === 'undefined':
        case environmentType === EnvironmentType.client && !poll:
            return 15000;
        case environmentType === EnvironmentType.server && !poll:
            return 500;

        default:
            return 1000;
    }
}

module.exports = new ApiMethod({
    name: 'trainTariffs2',
    params: {
        transportType: {
            type: 'String',
            required: true,
        },
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        now: {
            type: 'Number',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        flags: {
            type: 'Object',
            required: true,
        },
        startTime: {
            type: 'String',
        },
        endTime: {
            type: 'String',
        },
        poll: {
            type: 'Boolean',
            defaultValue: false,
        },
        environmentType: {
            type: 'String',
        },
    },
    action: (
        {
            transportType,
            pointFrom,
            pointTo,
            now,
            language,
            startTime,
            endTime,
            nationalVersion,
            poll,
            flags,
            environmentType,
        },
        req,
    ) => {
        if (!poll && isDirtyBot(req)) {
            return {
                ...defaultAnswer,
                isDirty: true,
            };
        }

        const path = `/${language}/api/segments/train-tariffs/${
            poll ? 'poll/' : ''
        }`;

        const query = {
            pointFrom,
            pointTo,
            // управление комиссией Яндекса
            experiment: Boolean(flags.__experiment),
            national_version: nationalVersion,
            includePriceFee: '1',
            partner: TrainPartner.im,
            useWizardSource: !poll,
            startTime,
            endTime,
        };

        return apiTimedGot(
            {
                ...apiConfig.purchaseOptions,
                timeout: getTimeout({environmentType, poll}),
                path,
                query,
                headers: getCommonHeaders({request: req, flags}),
                retries: environmentType === EnvironmentType.server ? 0 : 2,
                json: true,
            },
            req,
            {
                operationName: `/:language/api/segments/train-tariffs/${
                    poll ? 'poll/' : ''
                }`,
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                if (body.segments) {
                    body.segments = patchSegmentsFromBackend(body.segments)
                        .filter(segment => !shouldRemoveTariffs(segment, {now}))
                        .map(segment => {
                            if (transportType === SUBURBAN_TYPE) {
                                segment.canSupplySegments = false;
                            }

                            return segment;
                        });
                }

                return {trainTariffs: body};
            })
            .catch(err => {
                // Если инициализирующий запрос отвалился по таймауту, считаем, что все еще
                // может наладиться и попробуем заново инициализировать запрос с клиента (RASPFRONT-6374 и RASPFRONT-7187)
                if (
                    err.code === 'ETIMEDOUT' &&
                    environmentType === EnvironmentType.server &&
                    !poll
                ) {
                    return {
                        ...defaultAnswer,
                        timeoutError: true,
                    };
                }

                logger.error('server/api/trainTariffs2', err);

                return defaultAnswer;
            });
    },
});
