import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod} from 'bla';

import {ALL_TYPE} from '../../common/lib/transportType';

import TransferQueryStatus from '../../common/interfaces/transfer/TransferQueryStatus';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';
import requestIsFailed from './utils/requestIsFailed';
import {getCommonHeaders} from './utils/getCommonHeaders';
import {buildSegmentsFromTransfers} from '../../common/lib/segments/transfer/buildSegmentsFromTransfers';

const {api: apiConfig} = yaConfig();

const defaultAnswer = {
    transfers: [],
    status: TransferQueryStatus.done,
};

module.exports = new ApiMethod({
    name: 'transfersWithPrices',
    params: {
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        tld: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
        poll: {
            type: 'Boolean',
            required: true,
        },
        when: {
            type: 'String',
            required: true,
        },
        transportType: {
            type: 'String',
            defaultValue: ALL_TYPE,
        },
        flags: {
            type: 'Object',
        },
        isBot: {
            type: 'Boolean',
        },
    },
    action: async (
        {
            pointFrom,
            pointTo,
            when,
            transportType,
            tld,
            language,
            poll,
            flags,
            isBot,
        },
        req,
    ) => {
        try {
            const query = {
                pointFrom,
                pointTo,
                tld,
                when,
                language,
                isBot,
                includePriceFee: true,
                ...(transportType && transportType !== ALL_TYPE
                    ? {transportType}
                    : {}),
            };

            const {body} = await apiTimedGot(
                {
                    ...apiConfig.pathfinderOptions,
                    path: `/service/transfers-with-prices/${
                        poll ? 'poll/' : ''
                    }`,
                    query,
                    headers: getCommonHeaders({request: req, flags}),
                    json: true,
                },
                req,
                {
                    operationName: `/service/transfers-with-prices/${
                        poll ? 'poll/' : ''
                    }`,
                },
            );

            if (requestIsFailed(body)) {
                throw new Error(
                    `Request is failed: ${JSON.stringify(body.errors)}`,
                );
            }

            if (!body.transfer_variants) {
                // логируем случаи, когда не возвращаются пересадки
                logger.info(
                    'server/api/transfersWithPrices',
                    JSON.stringify(query),
                    JSON.stringify(body),
                );
            }

            const {transfer_variants = [], status} = body;

            const transfers = buildSegmentsFromTransfers({
                transfers: transfer_variants,
                language,
                transportType,
            });

            return {transfers, status};
        } catch (err) {
            logger.error('server/api/transfersWithPrices', err);

            return defaultAnswer;
        }
    },
});
