import {ApiMethod, ApiError} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {THREAD_TRANSPORT_TYPES} from '../../common/lib/transportType';

import {apiTimedGot} from '../helpers/timedGot';
import logger from '../logger';
import requestIsFailed from './utils/requestIsFailed';
import Error404 from '../../common/lib/errors/Error404';

const {api: apiConfig} = yaConfig();

module.exports = new ApiMethod({
    name: 'transportTypeOfThread',
    params: {
        threadId: {
            type: 'String',
            required: true,
        },
        language: {
            type: 'String',
            required: true,
        },
    },
    action: ({threadId, language}, req) => {
        const query = {
            mixed_uid: threadId,
        };

        return apiTimedGot(
            {
                ...apiConfig.options,
                path: `/${language}/thread/quick/`,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/thread/quick/',
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new ApiError(
                        null,
                        `Request failed: ${JSON.stringify(body)}`,
                    );
                }

                const transportType = body.result && body.result.transportType;

                if (!THREAD_TRANSPORT_TYPES.includes(transportType)) {
                    throw new ApiError(
                        null,
                        `Неизвестный тип транспорта для нитки. ${JSON.stringify(
                            body,
                        )}`,
                    );
                }

                return {transportType};
            })
            .catch(err => {
                // TODO: этот костылек можно будет удалить после того как morda-backend
                //  научится отдавать 404 для неизвестных ей ниток
                //  Можно смело удалять после 2018.04.01
                if (err.statusCode && err.statusCode !== 200) {
                    throw new Error404();
                }
                // конец костылька

                if (err.statusCode === 404) {
                    throw new Error404();
                }

                logger.error('server/api/transportTypeOfThread.js', err);

                throw err;
            });
    },
});
