import http from 'http';
import express from 'express';
import yaConfig from '@yandex-int/yandex-config';
import environment from '@yandex-int/yandex-environment';
import cookieParser from 'cookie-parser';
import yaCSP from '@yandex-int/express-yandex-csp';
import getExpressTld from '@yandex-int/express-tld';
import path from 'path';

import Environment from '../common/interfaces/Environment';
import WebpackMode from '../common/interfaces/WebpackMode';

import cartesianProduct from '../common/lib/cartesianProduct';
import expressRequestId from './middleware/expressRequestId';
import addTracerSpan from './middleware/addTracerSpan';
import userNetwork from './middleware/userNetwork';
import Error404 from '../common/lib/errors/Error404';
import page404 from '../common/routes/page404';
import patchHeaderXOriginUrl from './middleware/patchHeaderXOriginUrl';
import getStats from '../webpack/getStats';

const config = yaConfig();
const expressTld = getExpressTld();

// Basic application setup
const app = express();

app.use(expressRequestId);
app.use(userNetwork);
app.use(addTracerSpan);

app.set('env', environment);
app.enable('trust proxy');
app.disable('etag');
app.disable('x-powered-by');

app.use(patchHeaderXOriginUrl);

// Enable CSP (Content Security Policy)
app.use(cookieParser());
app.use(yaCSP(config.csp));

// Определяем req.tld
app.use(expressTld);

const isDev = environment === Environment.development;
const platformEnv = process.env.PLATFORM;
const modernBrowserEnv = process.env.MODERN_BROWSER;
const webpackModeEnv = process.env.WEBPACK_MODE || WebpackMode.development;

// Development setup: logging, static, etc.
if (isDev) {
    app.use(require('morgan')('short'));

    const webpack = require('webpack');
    const webpackConfig = require('../webpack/config').default;
    const webpackDevMiddleware = require('webpack-dev-middleware');

    const useWebpack = (platform, modernBrowser) => {
        app.use(
            webpackDevMiddleware(
                webpack(
                    webpackConfig({
                        platform,
                        modernBrowser,
                        mode: webpackModeEnv,
                    }),
                ),
                {
                    lazy: false,
                    stats: getStats(),
                },
            ),
        );
    };

    const variants = cartesianProduct(
        platformEnv ? [platformEnv] : ['mobile', 'desktop'],
        modernBrowserEnv ? [modernBrowserEnv === '1'] : [true, false],
    );

    variants.forEach(([platform, modernBrowser]) => {
        useWebpack(platform, modernBrowser);
    });

    app.use(express.static(path.join(__dirname, '../static')));
    app.use(express.static(path.join(__dirname, '../www')));
}

// Simple handlers that don't need middlewares go first
import {index as ping} from './controllers/ping';
import makeError from './controllers/makeError';
import searchInformer from './controllers/searchInformer';
import stationInformer from './controllers/stationInformer';

app.get('/ping', ping);
app.get('/promo/rasp/ping', ping);
app.get('/makeError', makeError);
app.get('/informers/v2/search/', searchInformer);
app.get('/informers/search/', searchInformer);
app.get('/informers/v2/station/:id/', stationInformer);
app.get('/informers/station/:id/', stationInformer);

import logger from './logger';
import api, {ApiError} from './api';

import bodyParser from 'body-parser';

app.use(bodyParser.json({limit: '10mb'}));

// Main middleware set
import oldMiddleware from './oldMiddleware';

oldMiddleware(app, config);

import serverRouter from './router';

serverRouter(app);

// Main app routing setup
import commonRouter from '../common/router';
import middleware from './middleware';
import injectToCreateStore from '../common/store';
import routerAdapter from './routerAdapter';
import errorRegisterForRootSpan from './middleware/errorRegisterForRootSpan';

const adapter = routerAdapter(app, injectToCreateStore, {
    api,
    ApiError,
    logger,
});

middleware(adapter);

commonRouter(adapter, {
    isServer: true,
});

// 404 handling
app.use((err, req, res, next) => {
    if (err instanceof Error404) {
        return adapter.wrap(page404)(req, res, next);
    }

    return next(err);
});

// error handing for opentracing
app.use(errorRegisterForRootSpan);

if (isDev) {
    /*
    Этот кусочек кода появился из-за того, что при локальной разработке
    браузер упорно пытается получить фавиконку с http://localhost/favicon.ico.
    При этом выкидывается ошибка детектинга языка и мне надоело наблюдать ее в консоли.
     */
    app.use((err, req, res, next) => {
        if (
            typeof req.tld !== 'undefined' &&
            req.tld === 'localhost' &&
            typeof req.path !== 'undefined' &&
            req.url === '/favicon.ico'
        ) {
            return res.status(404).send('Not found');
        }

        next(err);
    });
}

// eslint-disable-next-line @typescript-eslint/no-unused-vars
app.use((err, req, res, next) => {
    const status = err.statusCode || 500;
    const pathToFile = 'server/app';

    logger.error(pathToFile, err, {level: 'error'});

    if (res.finished) {
        return;
    }

    try {
        res.status(status).send(http.STATUS_CODES[status]);
    } catch (e) {
        logger.error(pathToFile, err, {
            message: `Failed to send ${status}`,
            tags: {http: status},
        });
        res.end('');
    }
});

export default app;
