import React from 'react';
import {Request, Response, NextFunction} from 'express';
import fsExtra from 'fs-extra';
import environment from '@yandex-int/yandex-environment';
import uniqBy from 'lodash/uniqBy';
import path from 'path';

import momentFunc from '../../reexports/moment-timezone';

import {
    DEFAULT_SIZE,
    INFORMER_MATCHED_COLORS,
    DEFAULT_COLOR_ID,
} from '../../common/lib/informer/constants';

import Req from '../../common/interfaces/router/Req';
import ISegment from '../../common/interfaces/segment/ISegment';
import {FilterTransportType} from '../../common/lib/transportType';
import IExpressRequest from '../../common/interfaces/IExpressRequest';
import GetParameters from '../../common/interfaces/lib/url/GetParameters';
import DateSpecialValue from '../../common/interfaces/date/DateSpecialValue';
import ISegmentFromApi from '../../common/interfaces/segment/ISegmentFromApi';
import SearchInformerReqQuery from '../../common/interfaces/router/SearchInformerReqQuery';
import InformerColor from '../../common/interfaces/components/informer/InformerColor';
import ISearchInformer from '../../common/interfaces/components/informer/ISearchInformer';
import SearchInformerSegments from '../../common/interfaces/components/informer/SearchInformerSegments';
import EnvironmentType from '../../common/interfaces/EnvironmentType';
import Environment from '../../common/interfaces/Environment';
import CurrencyCode from '../../common/interfaces/CurrencyCode';

import isPoint from '../../common/lib/point/isPoint';
import getInformerSvgSprite from '../helpers/getInformerSvgSprite';
import {getHostForInformerBundle} from '../../webpack/staticUtils';
import getValueFromEnum from '../../common/lib/enum/getValueFromEnum';
import getReactInformerBundle from '../helpers/getReactInformerBundle';
import getNationalVersion from '../../common/lib/url/getNationalVersion';
import getNationalLanguage from '../../common/lib/lang/getNationalLanguage';
import isValidInformerSize from '../../common/lib/informer/isValidInformerSize';
import patchSegments, {
    isGoneCheck,
} from '../../common/lib/segments/patchSegments';
import buildSegmentRenderingData from '../../common/lib/segments/buildSegmentRenderingData';

import api from '../../server/api';

function getColorFromId(id: number): InformerColor {
    return (
        INFORMER_MATCHED_COLORS[id] || INFORMER_MATCHED_COLORS[DEFAULT_COLOR_ID]
    );
}

// Для метасегмента вернет количество его подсегментов, для обычного сегмента вернет 1
function getSubSegmentsCount(segment: ISegmentFromApi): number {
    return segment.isMetaSegment ? segment.subSegments?.length || 0 : 1;
}

// Считает сколько в итоге будет элементов, учитывая что сегменты содержат мета-сегменты
function getCountSegmentsIncludingSubegments(
    segments: ISegmentFromApi[],
): number {
    return segments.reduce((count: number, segment: ISegmentFromApi) => {
        return (count += getSubSegmentsCount(segment));
    }, 0);
}

// Обрезает массив до нужной длины с учетом раскрытия мета-сегментов
function getNeededToShowSegments(
    segments: ISegmentFromApi[],
    size: number,
): ISegmentFromApi[] {
    let counter = 0;

    const index = segments.findIndex(segment => {
        counter += getSubSegmentsCount(segment);

        if (counter > size) {
            return true;
        }

        return false;
    });

    if (index < 0) {
        return segments;
    }

    return segments.slice(0, index);
}

// Определяет для каждого элемента строку определяющую уникальность сегмента
// и удаляет дубликаты
function getUniqSegmentsById(segments: ISegmentFromApi[]): ISegmentFromApi[] {
    const segmentsFromApiWithUniqId = segments.map(segment => {
        const {
            title,
            departure,
            stationFrom: {id: stationFromId},
            arrival,
            stationTo: {id: stationToId},
        } = segment;

        return {
            ...segment,
            id: `${title}_${departure}_${stationFromId}_${arrival}_${stationToId}`,
        };
    });

    return uniqBy(segmentsFromApiWithUniqId, 'id');
}

// Раскрывает мета-мегменты и обрезает список до размера информера
function getDisplayedSegments(
    segments: ISegment[],
    size: number,
): SearchInformerSegments {
    return segments
        .reduce(
            (displayedSegments: SearchInformerSegments, segment: ISegment) => {
                if (segment.subSegments?.length) {
                    displayedSegments.push(...segment.subSegments);
                } else {
                    displayedSegments.push(segment);
                }

                return displayedSegments;
            },
            [],
        )
        .slice(0, size);
}

export default async function searchInformer(
    req: Request &
        Pick<IExpressRequest, 'tld'> &
        Pick<IExpressRequest, 'nonce'>,
    res: Response,
    next: NextFunction,
): Promise<void> {
    try {
        const query = (req.query ||
            {}) as GetParameters<SearchInformerReqQuery>;
        const {
            color: colorIdFromQuery,
            size: sizeFromQuery,
            fromId: fromIdFromQuery,
            toId: toIdFromQuery,
            type: typeFromQuery,
        } = query;

        const parsedSize =
            typeof sizeFromQuery === 'string'
                ? parseInt(sizeFromQuery, 10)
                : DEFAULT_SIZE;
        const size = isValidInformerSize(parsedSize)
            ? parsedSize
            : DEFAULT_SIZE;
        const type =
            (typeof typeFromQuery === 'string' &&
                getValueFromEnum(typeFromQuery, FilterTransportType)) ||
            FilterTransportType.all;
        const parsedColorId =
            typeof colorIdFromQuery === 'string'
                ? parseInt(colorIdFromQuery, 10)
                : DEFAULT_COLOR_ID;
        const colorId = !isNaN(parsedColorId)
            ? parsedColorId
            : DEFAULT_COLOR_ID;
        const fromId =
            typeof fromIdFromQuery === 'string' ? fromIdFromQuery : undefined;
        const toId =
            typeof toIdFromQuery === 'string' ? toIdFromQuery : undefined;

        const tld = req.tld;
        const language = getNationalLanguage(tld);

        const svgSprite = getInformerSvgSprite({language});
        const hostForInformerBundle = getHostForInformerBundle({language});
        const {RootInformers, ReactDOMServer, moment} =
            getReactInformerBundle(language);

        moment.locale(language);

        res.setHeader('Content-Type', 'text/html; charset=utf-8');

        if (!fromId || !toId || !isPoint(fromId) || !isPoint(toId)) {
            throw new Error('Одна или обе точки указаны неверно');
        }

        const [now, parseContext] = await Promise.all([
            api.execTimeCorrection({}, req as Req),
            api.execParseContext(
                {
                    tld,
                    language,
                    transportType: type,
                    fromKey: fromId,
                    toKey: toId,
                },
                req as Req,
            ),
        ]);

        const {from: parseContextFrom, to: parseContextTo} = parseContext;

        const when = {date: DateSpecialValue.today};
        const searchParams = {
            context: {
                transportType: type,
                from: {key: parseContextFrom.key},
                to: {key: parseContextTo.key},
                when,
                language,
            },
            nationalVersion: getNationalVersion(tld),
        };

        const {
            search: {segments: segmentsFromSearch, context},
        } = await api.execSearch(searchParams, req as Req);

        const nowMoment = momentFunc(now);

        const notGoneSegments = segmentsFromSearch.filter(
            segment => !isGoneCheck({}, nowMoment, segment.departureLocalDt),
        );

        const countSegments =
            getCountSegmentsIncludingSubegments(notGoneSegments);

        let segmentsFromApi = notGoneSegments;

        if (countSegments < size) {
            const {
                search: {segments: tomorrowSegmentsFromBackend},
            } = await api.execSearch(
                {
                    ...searchParams,
                    context: {
                        ...searchParams.context,
                        when: {date: DateSpecialValue.tomorrow},
                    },
                },
                req as Req,
            );

            segmentsFromApi = getUniqSegmentsById([
                ...segmentsFromApi,
                ...tomorrowSegmentsFromBackend,
            ]);
        }

        const neededToShowSegments = getNeededToShowSegments(
            segmentsFromApi,
            size,
        );

        const patchSegmentsParams = {
            segments: neededToShowSegments,
            meta: {
                tld,
                context: {
                    time: {
                        now,
                    },
                },
                environment: EnvironmentType.server,
                isProduction: environment === Environment.production,
                currencies: {
                    nationalCurrency: CurrencyCode.rub,
                    preferredCurrency: CurrencyCode.rub,
                    availableCurrencies: [],
                    currencyRates: {},
                },
            },
        };

        const patchedSegments = patchSegments(patchSegmentsParams);

        const displayedSegments = getDisplayedSegments(
            patchedSegments,
            size,
        ).map(segment => buildSegmentRenderingData(segment));

        momentFunc.locale(language);
        const dateMomentOnStationFrom = displayedSegments.length
            ? momentFunc.parseZone(displayedSegments[0].departureLocalDt)
            : momentFunc(now).tz(parseContext.from.timezone || '');

        const dataForSearchInformer: ISearchInformer = {
            tld,
            language,
            transportType: type,
            segments: displayedSegments,
            color: getColorFromId(colorId),
            size,
            pointFrom: context.isChanged
                ? context.search.pointFrom
                : context.original.pointFrom,
            pointTo: context.isChanged
                ? context.search.pointTo
                : context.original.pointTo,
            dateMomentOnStationFrom,
        };

        const appString = ReactDOMServer.renderToStaticMarkup(
            <RootInformers
                svgSprite={svgSprite}
                hostForInformerBundle={hostForInformerBundle}
                dataForSearchInformer={dataForSearchInformer}
                nonce={req.nonce}
            />,
        );

        res.send(`<!DOCTYPE html>${appString}`);
    } catch (e) {
        try {
            const content = await fsExtra.readFile(
                path.join(__dirname, '../../www/404.ru.html'),
                {encoding: 'utf8'},
            );

            res.status(404);
            res.send(content);
        } catch (err) {
            next(e);
        }
    }
}
