import React from 'react';
import {Request, Response, NextFunction} from 'express';
import fsExtra from 'fs-extra';
import path from 'path';
import uniqBy from 'lodash/uniqBy';

import {
    DEFAULT_SIZE,
    DEFAULT_COLOR_ID,
    INFORMER_MATCHED_COLORS,
} from '../../common/lib/informer/constants';
import {DIRECTION_CODE_ALL} from '../../common/lib/station/stationConstants';

import Req from '../../common/interfaces/router/Req';
import DateRobot from '../../common/interfaces/date/DateRobot';
import {TransportType} from '../../common/lib/transportType';
import DateMoment from '../../common/interfaces/date/DateMoment';
import IExpressRequest from '../../common/interfaces/IExpressRequest';
import ITerminal from '../../common/interfaces/state/station/ITerminal';
import GetParameters from '../../common/interfaces/lib/url/GetParameters';
import StationType from '../../common/interfaces/state/station/StationType';
import IThreadPlane from '../../common/interfaces/state/station/IThreadPlane';
import IApiStationRequest from '../../common/interfaces/api/IApiStationRequest';
import AllThreadType from '../../common/interfaces/state/station/AllThreadType';
import StationSubtype from '../../common/interfaces/state/station/StationSubtype';
import IThreadRailroad from '../../common/interfaces/state/station/IThreadRailroad';
import IStationCompany from '../../common/interfaces/state/station/IStationCompany';
import InformerColor from '../../common/interfaces/components/informer/InformerColor';
import StationEventList from '../../common/interfaces/state/station/StationEventList';
import InformerTheme from '../../common/interfaces/components/informer/InformerTheme';
import StationDateSpecialValue from '../../common/interfaces/date/StationDateSpecialValue';
import IStationInformer from '../../common/interfaces/components/informer/IStationInformer';
import StationInformerReqQuery from '../../common/interfaces/router/StationInformerReqQuery';
import StationCompaniesById from '../../common/interfaces/state/station/StationCompaniesById';
import StationInformerReqParams from '../../common/interfaces/router/StationInformerReqParams';
import IStationScheduleBlock from '../../common/interfaces/state/station/IStationScheduleBlock';
import StationInformerType from '../../common/interfaces/components/informer/StationInformerType';
import AllStationInformerScheduleThreadType from '../../common/interfaces/components/informer/AllStationInformerScheduleThreadType';

import getInformerSvgSprite from '../helpers/getInformerSvgSprite';
import isPlaneThreads from '../../common/lib/station/isPlaneThreads';
import {getHostForInformerBundle} from '../../webpack/staticUtils';
import getValueFromEnum from '../../common/lib/enum/getValueFromEnum';
import getReactInformerBundle from '../helpers/getReactInformerBundle';
import isScheduleThread from '../../common/lib/station/isScheduleThread';
import isRailroadThreads from '../../common/lib/station/isRailroadThreads';
import getNationalLanguage from '../../common/lib/lang/getNationalLanguage';
import isValidInformerSize from '../../common/lib/informer/isValidInformerSize';
import getTimeFromDateMoment from '../../common/lib/date/getTimeFromDateMoment';
import isScheduleIntervalThread from '../../common/lib/station/isScheduleIntervalThread';
import getDateRobotFromDateMoment from '../../common/lib/date/getDateRobotFromDateMoment';
import isStationInformerScheduleThread from '../../common/lib/informer/isStationInformerScheduleThread';
import isAllStationInformerScheduleThreads from '../../common/lib/informer/isAllStationInformerScheduleThreads';
import isStationInformerScheduleIntervalThread from '../../common/lib/informer/isStationInformerScheduleIntervalThread';

import api from '../../server/api';

function getColorFromId(id: number): InformerColor {
    return (
        INFORMER_MATCHED_COLORS[id] || INFORMER_MATCHED_COLORS[DEFAULT_COLOR_ID]
    );
}

function getNotGoneThreads(
    threads: AllThreadType[] | AllStationInformerScheduleThreadType[],
    now: DateMoment,
): AllThreadType[] | AllStationInformerScheduleThreadType[] {
    if (isAllStationInformerScheduleThreads(threads)) {
        return threads.filter(thread => {
            if (isStationInformerScheduleThread(thread)) {
                const {departureFrom} = thread;

                if (departureFrom < now) {
                    return false;
                }

                return true;
            }

            if (isStationInformerScheduleIntervalThread(thread)) {
                const {beginTime, endTime} = thread;

                const nowTime = getTimeFromDateMoment(now);

                if (!nowTime) {
                    return false;
                }

                if (beginTime <= nowTime && endTime >= nowTime) {
                    return true;
                }

                return false;
            }

            return false;
        });
    }

    if (isRailroadThreads(threads) || isPlaneThreads(threads)) {
        return (threads as (IThreadPlane | IThreadRailroad)[]).filter(
            thread => {
                const {
                    eventDt: {datetime},
                } = thread;

                const threadIsGone = datetime && datetime < now;

                return !threadIsGone;
            },
        );
    }

    return threads;
}

function getThreadsFromScheduleBlocks(
    scheduleBlocks: IStationScheduleBlock[],
    whenDate?: DateRobot,
): AllStationInformerScheduleThreadType[] {
    const threads: AllStationInformerScheduleThreadType[] = [];

    scheduleBlocks.forEach(scheduleBlock => {
        const {comment, number, title, transportType, schedule} = scheduleBlock;

        schedule.forEach(stationSchelule => {
            const {threads: scheduleThreads} = stationSchelule;

            scheduleThreads.forEach(thread => {
                const {canonicalUid} = thread;

                if (isScheduleIntervalThread(thread)) {
                    const {
                        interval: {endTime, beginTime, density},
                    } = thread;

                    threads.push({
                        comment,
                        number,
                        title,
                        transportType,
                        canonicalUid,
                        endTime,
                        beginTime,
                        density,
                        date: whenDate,
                    });
                }

                if (isScheduleThread(thread)) {
                    const {departureFrom} = thread;

                    threads.push({
                        comment,
                        number,
                        title,
                        transportType,
                        canonicalUid,
                        departureFrom,
                    });
                }
            });
        });
    });

    return threads.sort((prevThread, thread) => {
        const prevThreadDatetime =
            'departureFrom' in prevThread
                ? `${getDateRobotFromDateMoment(
                      prevThread.departureFrom,
                  )} ${getTimeFromDateMoment(prevThread.departureFrom)}`
                : `${prevThread.date} ${prevThread.beginTime}`;

        const threadDatetime =
            'departureFrom' in thread
                ? `${getDateRobotFromDateMoment(
                      thread.departureFrom,
                  )} ${getTimeFromDateMoment(thread.departureFrom)}`
                : `${thread.date} ${thread.beginTime}`;

        return prevThreadDatetime < threadDatetime
            ? -1
            : prevThreadDatetime > threadDatetime
            ? 1
            : 0;
    });
}

// Только нужные поля интерфейса IStationFromApi
interface IStationFromApiForGetPlaneAndRailroadStationInfo {
    threads: AllThreadType[];
    companies: IStationCompany[];
    now: DateMoment;
    terminals: ITerminal[];
}

// Только нужные поля интерфейса IStationFromApi
interface IStationFromApiForGetStationScheduleInfo {
    threads: AllStationInformerScheduleThreadType[];
    companies: IStationCompany[];
    now: DateMoment;
    terminals: ITerminal[];
}

interface IGetStationInfoForDisplay {
    size: number;
    tomorrowStationParams: IApiStationRequest; // Параметры чтобы дернуть ручку на завтрашний день, если на сегодняшний рейсов недостаточно
    req: Req;

    transportTypeForFilterThreads?: TransportType;
}

interface IGetPlaneAndRailroadStationInfoForDisplay
    extends IGetStationInfoForDisplay {
    stationDataFromApi: IStationFromApiForGetPlaneAndRailroadStationInfo;
}

interface IGetScheduleStationInfoForDisplay extends IGetStationInfoForDisplay {
    stationDataFromApi: IStationFromApiForGetStationScheduleInfo;
}

interface IStationInfoForDisplay {
    threads: AllThreadType[] | AllStationInformerScheduleThreadType[];
    companies: IStationCompany[];
    terminals: ITerminal[];
}

async function getPlaneAndRailroadStationInfoForDisplay({
    stationDataFromApi,
    size,
    tomorrowStationParams,
    req,

    transportTypeForFilterThreads,
}: IGetPlaneAndRailroadStationInfoForDisplay): Promise<IStationInfoForDisplay> {
    const {threads, companies, now, terminals} = stationDataFromApi;

    const threadsForDisplay: AllThreadType[] = [];
    const companiesForDisplay: IStationCompany[] = companies;
    const terminalsForDisplay: ITerminal[] = terminals;

    const notGoneThreads = getNotGoneThreads(threads, now);
    const notScheduleNotGoneThreads = !isAllStationInformerScheduleThreads(
        notGoneThreads,
    )
        ? notGoneThreads
        : [];

    if (notScheduleNotGoneThreads.length < size) {
        threadsForDisplay.push(...notScheduleNotGoneThreads);

        const tomorrowStationData = await api.execStation(
            {
                ...tomorrowStationParams,
            },
            req as Req,
        );

        const {
            threads: tomorrowThreads,
            companies: tomorrowCompanies,
            terminals: tomorrowTerminals,
        } = tomorrowStationData;

        threadsForDisplay.push(
            ...tomorrowThreads
                .filter(
                    (t: AllThreadType) => !isThreadExists(t, threadsForDisplay),
                )
                .slice(0, size - notScheduleNotGoneThreads.length),
        );
        companiesForDisplay.push(...tomorrowCompanies);
        terminalsForDisplay.push(...tomorrowTerminals);
    } else {
        threadsForDisplay.push(...notScheduleNotGoneThreads.slice(0, size));
    }

    return {
        threads: transportTypeForFilterThreads
            ? threadsForDisplay.filter(
                  thread =>
                      thread.transportType === transportTypeForFilterThreads,
              )
            : threadsForDisplay,
        companies: uniqBy(companiesForDisplay, 'id'),
        terminals: uniqBy(terminalsForDisplay, 'id'),
    };
}

function isThreadExists(
    thread: AllThreadType,
    threads: AllThreadType[],
): boolean {
    return threads.some(
        (t: AllThreadType) =>
            t.number === thread.number &&
            t.eventDt.datetime === thread.eventDt.datetime,
    );
}

async function getScheduleStationInfoForDisplay({
    stationDataFromApi,
    size,
    tomorrowStationParams,
    req,

    transportTypeForFilterThreads,
}: IGetScheduleStationInfoForDisplay): Promise<IStationInfoForDisplay> {
    const {threads, companies, now} = stationDataFromApi;

    const threadsForDisplay: AllStationInformerScheduleThreadType[] = [];
    const companiesForDisplay: IStationCompany[] = companies;

    const notGoneThreads = getNotGoneThreads(threads, now);
    const scheduleNotGoneThreads = isAllStationInformerScheduleThreads(
        notGoneThreads,
    )
        ? notGoneThreads
        : [];

    if (notGoneThreads.length < size) {
        threadsForDisplay.push(...scheduleNotGoneThreads);

        const tomorrowStationData = await api.execStation(
            {
                ...tomorrowStationParams,
            },
            req as Req,
        );

        const {
            companies: tomorrowCompanies,
            scheduleBlocks,
            whenDate,
        } = tomorrowStationData;

        const tomorrowThreads = getThreadsFromScheduleBlocks(
            scheduleBlocks,
            whenDate,
        );

        threadsForDisplay.push(
            ...tomorrowThreads.slice(0, size - scheduleNotGoneThreads.length),
        );
        companiesForDisplay.push(...tomorrowCompanies);
    } else {
        threadsForDisplay.push(...scheduleNotGoneThreads.slice(0, size));
    }

    return {
        threads: transportTypeForFilterThreads
            ? threadsForDisplay.filter(
                  thread =>
                      thread.transportType === transportTypeForFilterThreads,
              )
            : threadsForDisplay,
        companies: uniqBy(companiesForDisplay, 'id'),
        terminals: [],
    };
}

function getSubtypeFromInformerType(
    informerType: StationInformerType,
): StationSubtype | undefined {
    switch (informerType) {
        case StationInformerType.suburban:
            return StationSubtype.suburban;
        case StationInformerType.train:
            return StationSubtype.train;
        case StationInformerType.allTransport:
            return StationSubtype.tablo;

        default:
            return undefined;
    }
}

function getTransportTypeForFilterThreads(
    typeFromQuery: string,
): TransportType | undefined {
    switch (typeFromQuery) {
        case TransportType.bus:
            return TransportType.bus;
        case TransportType.water:
            return TransportType.water;
        case TransportType.train:
            return TransportType.train;
        case TransportType.suburban:
            return TransportType.suburban;

        default:
            return undefined;
    }
}

export default async function stationInformer(
    req: Request &
        Pick<IExpressRequest, 'tld'> &
        Pick<IExpressRequest, 'nonce'>,
    res: Response,
    next: NextFunction,
): Promise<void> {
    try {
        const query = (req.query ||
            {}) as GetParameters<StationInformerReqQuery>;

        const {id: stationIdFromParams} =
            req.params as StationInformerReqParams;
        const {
            color: colorIdFromQuery,
            size: sizeFromQuery,
            type: typeFromQuery,
            theme: themeFromQuery,
            event: eventFromQuery,
        } = query;

        const stationId = parseInt(stationIdFromParams, 10);
        const parsedSize =
            typeof sizeFromQuery === 'string'
                ? parseInt(sizeFromQuery, 10)
                : DEFAULT_SIZE;
        const size = isValidInformerSize(parsedSize)
            ? parsedSize
            : DEFAULT_SIZE;
        const informerType =
            (typeof typeFromQuery === 'string' &&
                getValueFromEnum(typeFromQuery, StationInformerType)) ||
            StationInformerType.allTransport;
        const parsedColorId =
            typeof colorIdFromQuery === 'string'
                ? parseInt(colorIdFromQuery, 10)
                : DEFAULT_COLOR_ID;
        const colorId = !isNaN(parsedColorId)
            ? parsedColorId
            : DEFAULT_COLOR_ID;
        const theme =
            typeof themeFromQuery === 'string'
                ? getValueFromEnum(themeFromQuery, InformerTheme)
                : undefined;
        const event =
            typeof eventFromQuery === 'string'
                ? getValueFromEnum(eventFromQuery, StationEventList)
                : StationEventList.departure;

        const transportTypeForFilterThreads =
            typeof typeFromQuery === 'string'
                ? getTransportTypeForFilterThreads(typeFromQuery)
                : undefined;

        const tld = req.tld;
        const language = getNationalLanguage(tld);

        const svgSprite = getInformerSvgSprite({language});
        const hostForInformerBundle = getHostForInformerBundle({language});
        const {RootInformers, ReactDOMServer, moment} =
            getReactInformerBundle(language);

        moment.locale(language);

        res.setHeader('Content-Type', 'text/html; charset=utf-8');

        const subtype = getSubtypeFromInformerType(informerType);

        const stationParams = {
            stationId,
            isMobile: false,
            language,

            subtype,
            event,
            direction: DIRECTION_CODE_ALL,
        };

        const stationDataFromApi = await api.execStation(
            stationParams,
            req as Req,
        );

        const tomorrowStationParams = {
            ...stationParams,

            date: StationDateSpecialValue.tomorrow,
        };

        let stationInfoForDisplay: IStationInfoForDisplay;

        const stationType = stationDataFromApi.type;
        const isScheduleStationType =
            stationType === StationType.bus ||
            stationType === StationType.water;

        if (isScheduleStationType) {
            const threads = getThreadsFromScheduleBlocks(
                stationDataFromApi.scheduleBlocks,
                stationDataFromApi.whenDate,
            );

            stationInfoForDisplay = await getScheduleStationInfoForDisplay({
                stationDataFromApi: {
                    ...stationDataFromApi,
                    threads,
                },
                size,
                tomorrowStationParams,
                req: req as Req,

                transportTypeForFilterThreads,
            });
        } else {
            stationInfoForDisplay =
                await getPlaneAndRailroadStationInfoForDisplay({
                    stationDataFromApi,
                    tomorrowStationParams,
                    size,
                    req: req as Req,

                    transportTypeForFilterThreads,
                });
        }

        const {
            id,
            type,
            title,
            now,
            event: eventFromApi,
            trusted,

            whenDate,
            currentSubtype,
            fullTitle,
        } = stationDataFromApi;

        const dataForStationInformer: IStationInformer = {
            tld,
            language,
            stationData: {
                id,
                type,
                title,
                now,
                event: isScheduleStationType
                    ? StationEventList.departure
                    : eventFromApi,
                threads: stationInfoForDisplay.threads,
                companiesById:
                    stationInfoForDisplay.companies.reduce<StationCompaniesById>(
                        (result, company) => {
                            result[company.id] = company;

                            return result;
                        },
                        {},
                    ),
                terminals: stationInfoForDisplay.terminals,
                trusted,

                fullTitle,
                subtype: currentSubtype,
                whenDate,
            },
            size,

            color: getColorFromId(colorId),
            theme,
            informerType,
        };

        const appString = ReactDOMServer.renderToStaticMarkup(
            <RootInformers
                svgSprite={svgSprite}
                hostForInformerBundle={hostForInformerBundle}
                dataForStationInformer={dataForStationInformer}
                nonce={req.nonce}
            />,
        );

        res.send(`<!DOCTYPE html>${appString}`);
    } catch (e) {
        try {
            const content = await fsExtra.readFile(
                path.join(__dirname, '../../www/404.ru.html'),
                {encoding: 'utf8'},
            );

            res.status(404);
            res.send(content);
        } catch (err) {
            next(e);
        }
    }
}
