import {apiTimedGot} from './timedGot';
import logger from '../logger';
import kebabCase from 'lodash/kebabCase';
import yaConfig from '@yandex-int/yandex-config';
import {ApiMethod, ApiError} from 'bla';

const {api: apiConfig} = yaConfig();

export const createApiMethod = method => {
    const {
        name,
        params,
        buildPath,
        buildOperationName,
        buildQuery = () => ({}),
        requestConfigOptions = apiConfig.options,
        getTimeout = () => {
            if (requestConfigOptions.timeout) {
                return requestConfigOptions.timeout;
            }

            return 5000;
        },
        defaultResponse,

        logPrefix = `server/api/${name}`,

        onSuccess = () => res => res.body,

        onError = () => err => {
            logger.error(logPrefix, err);

            if (defaultResponse !== undefined) {
                return defaultResponse;
            }

            if (err instanceof ApiError) {
                throw err;
            }

            let message = err.message || '';

            if (err.response && err.response.body) {
                message += `${message ? ' ' : ''}ResponseBody: ${
                    err.response.body
                }.`;
            }

            const type = err.code === 'ETIMEDOUT' ? ApiError.TIMEOUT : null;

            const error = new ApiError(type, message);

            // прокидываем statusCode ошибки, чтобы можно было корректно ее обработать
            if (err.statusCode) {
                error.statusCode = err.statusCode;
            }

            throw error;
        },

        makeRequest = (requestParams, req) =>
            apiTimedGot(
                {
                    ...requestConfigOptions,
                    timeout: getTimeout(requestParams, req),
                    json: true,
                    path: buildPath(requestParams, req),
                    query: buildQuery(requestParams, req),
                },
                req,
                {
                    operationName: buildOperationName
                        ? buildOperationName()
                        : buildPath(requestParams, req),
                },
            ),

        action = (...args) =>
            makeRequest(...args)
                .then(onSuccess(...args))
                .catch(onError(...args)),
    } = method;

    return new ApiMethod({name, params, action});
};

export const createSettlementApiMethod = name =>
    createApiMethod({
        name,
        params: {
            settlementId: {
                type: 'Number',
                required: true,
            },
            language: {
                type: 'String',
                required: true,
            },
        },
        buildPath: ({language, settlementId}) =>
            `/${language}/settlement/${settlementId}/${kebabCase(name)}/`,
        buildOperationName: () =>
            `/:language/settlement/:settlementId/${kebabCase(name)}/`,
    });

export const createNoopApiMethod = (name, params = {}, answer = {}) =>
    new ApiMethod({
        name,
        params,
        action: () => {
            logger.error(
                'server/helpers/api',
                new ApiError(
                    null,
                    `noop method with name "${name}" was called`,
                ),
            );

            return answer;
        },
    });

export const handleError = (err, apiPath) => {
    logger.error(apiPath, err);
};
