export default function cookies(req, res) {
    return {
        get(name) {
            return req.cookies[name];
        },

        set(name, value, options = {}) {
            res.cookie(name, value, options);

            setCookie(res, name, value, options);
        },

        expire(name, options = {}) {
            res.clearCookie(name, options);

            deleteCookie(res, name);
        },
    };
}

export function createCookiesIfNotExists({req, res}) {
    if (req && !req.cookies) {
        res.cookies = {};
    }

    if (res) {
        if (!res.cookies) {
            res.cookies = {};
        }

        if (!res.deletedCookies) {
            res.deletedCookies = [];
        }
    }
}

export function deleteCookie(res, name) {
    createCookiesIfNotExists({res});

    delete res.cookies[name];
    res.deletedCookies.push(name);
}

export function setCookie(res, name, value, options = {}) {
    createCookiesIfNotExists({res});

    if ('maxAge' in options) {
        options.expires = new Date(Date.now() + options.maxAge);
    }

    if (options.expires && options.expires < Date.now()) {
        deleteCookie(res, name);
    } else {
        res.cookies[name] = value;

        res.deletedCookies = res.deletedCookies.filter(
            nameCookie => nameCookie !== name,
        );
    }
}
