/**
 * @typedef {number} Percentage
 * Сколько процентов пользователей попадает в эксперимент. Число от 0 до 100.
 */

/**
 * @typedef {string} Key
 * Ключ эксперимента. По соглашению должен начинаться с '__'.
 */

/**
 * @typedef {(boolean|number|string)} Value
 * Значение эксперимента.
 */

/**
 * @typedef {function} Type
 * Тип значения эксперимента: Boolean, String или Number.
 */

/**
 * @typedef {Object} ValuePercentage
 * Описание допустимого значения эксперимента
 *
 * @property {Value} value
 * @property {Percentage} percentage
 */

/**
 * @typedef {Object.<Key, Value[]>} Dependencies
 * Описание зависимостей эксперимента.
 */

/**
 * @typedef {Object} Experiment
 * Описание эксперимента
 *
 * @property {Type} type
 * @property {Percentage} [percentage] - обязательно для Boolean
 * @property {ValuePercentage[]} [values] - обязательно для String и Number
 * @property {Value} [defaultValue] - обязательно для String и Number
 * @property {Dependencies} [dependencies]
 */

/**
 * Генерирует имя экспериментальной куки
 *
 * @param {Key} key
 *
 * @return {string}
 */
export function generateCookieName(key) {
    return `experiment${key}`;
}

/**
 * Генерирует новое случайное значение для эксперимента.
 *
 * @param {ValuePercentage[]} values
 * @param {Value} defaultValue
 *
 * @return {Value}
 */
export function generateExperimentValue({values, defaultValue}) {
    const rnd = Math.random() * 100;

    let percentage = 0;

    for (let i = values.length - 1; i >= 0; i--) {
        percentage += values[i].percentage;

        if (rnd < percentage) {
            return values[i].value;
        }
    }

    return defaultValue;
}

/**
 * Получает значение эксперимента нужного типа из строки
 *
 * @param {string} rawValue
 * @param {Type} type
 * @param {ValuePercentage[]} values
 * @param {Value} defaultValue
 *
 * @return {Value}
 */
export function parseRawValue(rawValue, {type, values, defaultValue}) {
    const parsedValue = type(rawValue);

    if (values.find(({value}) => value === parsedValue)) {
        return parsedValue;
    }

    return defaultValue;
}

/**
 * Сериализует значение эксперимента в строку
 *
 * @param {Value} value
 * @param {Type} type
 *
 * @return {string}
 */
export function serializeExperimentValue(value, {type}) {
    switch (type) {
        case Boolean:
            return value ? '1' : '';

        default:
            return String(value);
    }
}

/**
 * Дополняет краткую декларацию эксперимента (тип Boolean) для удобства вычислений
 *
 * @param {Experiment} experiment
 *
 * @return {Experiment}
 */
export function completeExperimentData(experiment) {
    switch (experiment.type) {
        case Boolean: {
            const defaultValue = Boolean(experiment.defaultValue);

            return {
                ...experiment,
                values: [
                    {value: !defaultValue, percentage: experiment.percentage},
                ],
                defaultValue,
            };
        }

        default:
            return experiment;
    }
}

/**
 * Возвращает значение экспериментальной куки у пользователя.
 * Генерирует и устанавливает, если куки нет.
 *
 * @param {Key} key
 * @param {Experiment} experiment
 * @param {Object} cookies
 *
 * @return {Value}
 */
export function getOrSetExperimentCookie(key, experiment, cookies) {
    const cookieName = generateCookieName(key);
    const cookieRawValue = cookies.get(cookieName);

    if (cookieRawValue !== undefined && !experiment.dynamic) {
        return parseRawValue(cookieRawValue, experiment);
    }

    const value = generateExperimentValue(experiment);
    const cookieValue = serializeExperimentValue(value, experiment);

    cookies.set(cookieName, cookieValue, {
        expires: new Date(Date.now() + 3600000 * 24 * 30), // month
    });

    return value;
}

/**
 * Сортирует ключи экспериментов так чтобы каждый эксперимент шел строго после своих зависимостей.
 *
 * @param {Experiment} experiments
 *
 * @return {Key[]}
 */
export function sortExperimentKeysByDeps(experiments) {
    const sortedExperimentKeys = [];

    function collectDependencies(key) {
        if (sortedExperimentKeys.includes(key)) {
            return;
        }

        const experiment = experiments[key];
        const experimentDependencies = experiment.dependencies;

        if (experimentDependencies) {
            Object.keys(experimentDependencies)
                .sort()
                .forEach(collectDependencies);
        }

        sortedExperimentKeys.push(key);
    }

    Object.keys(experiments).sort().forEach(collectDependencies);

    return sortedExperimentKeys;
}

/**
 * Проверяет, выполняются ли зависимости на заданном наборе значений
 *
 * @param {Dependencies} deps
 * @param {Object.<Key, Value>} flags
 *
 * @return {boolean}
 */
export function meetExperimentDependencies(deps, flags) {
    return Object.keys(deps).every(depKey =>
        deps[depKey].includes(flags[depKey]),
    );
}
