import yaConfig from '@yandex-int/yandex-config';
import environment from '@yandex-int/yandex-environment';

import Environment from '../../common/interfaces/Environment';

import logger from '../logger';

const config = yaConfig();

const USE_HTTP_SERVICES = process.env.USE_HTTP_SERVICES;

const promisifySyncGeobase = yandexGeobase => {
    const methods = ['chiefRegionId'];

    const promisified = {
        __syncGeobase__: true,
        parents: (...args) =>
            Promise.resolve(yandexGeobase.getParentsIds(...args)),
        regionById: (...args) =>
            Promise.resolve(yandexGeobase.getRegionById(...args)),
        traitsByIp: (...args) =>
            Promise.resolve(yandexGeobase.getIpTraits(...args)),
        linguisticsForRegion: (...args) =>
            Promise.resolve(yandexGeobase.getLinguistics(...args)),
        // eslint-disable-next-line camelcase
        pinpointGeolocation: ({yandex_gid, x_forwarded_for, x_real_ip}) =>
            Promise.resolve(
                yandexGeobase.getRegion(
                    String(yandex_gid),
                    x_forwarded_for,
                    String(x_real_ip),
                ),
            ),
    };

    methods.forEach(method => {
        promisified[method] = function (...args) {
            return Promise.resolve().then(() => yandexGeobase[method](...args));
        };
    });

    return promisified;
};

let geobase;

if (USE_HTTP_SERVICES) {
    const HttpGeobase = require('http-geobase').default;

    geobase = new HttpGeobase(
        config.httpGeobase.server,
        config.httpGeobase.clientOptions,
    );
} else {
    geobase = promisifySyncGeobase(
        require('@yandex-int/yandex-geobase').default.v6(config.geobase),
    );
}

const logError = method => err => {
    logger.error(method, err);

    return null;
};

export function getUserNetworkTraits(req) {
    if (environment === Environment.development && !req.headers['x-real-ip']) {
        req.headers['x-real-ip'] = config.mockRealIpX;
    }

    if (!req.headers['x-real-ip']) {
        return Promise.resolve(null);
    }

    return (
        Promise.resolve()
            .then(() => geobase.traitsByIp(req.headers['x-real-ip']))
            // всегда резолвить промис, как с синхронной геобазой
            .catch(logError('getUserNetworkTraits', req))
    );
}

// Получает гео-идентифиактор текущего места по заголовкам HTTP-запроса
export function getGeoIdByRequest(req) {
    const params = {
        x_forwarded_for: req.ips.join(','),
        x_real_ip: req.header('x-real-ip') || req.ip,
        yandex_gid:
            req.cookies.yandex_gid && parseInt(req.cookies.yandex_gid, 10),
        yp_cookie: req.cookies.yp,
        ys_cookie: req.cookies.ys,
        is_trusted: true, // WARNING: this is to support old nodejs binding behaviour (region method) and thus provide compatibility and seamless migration
        allow_yandex: true,
    };

    return (
        Promise.resolve()
            .then(() => geobase.pinpointGeolocation(params))
            .then(res => {
                if (geobase.__syncGeobase__) {
                    return res;
                }

                return res.region_id;
            })
            // всегда резолвить промис, как с синхронной геобазой
            .catch(logError('getGeoIdByRequest', req))
    );
}

export function getRegionByRequest(req) {
    return this.getGeoIdByRequest(req)
        .then(geoId => {
            let region;

            return geobase
                .regionById(geoId)
                .then(_region => {
                    region = _region;

                    return geobase.linguisticsForRegion(
                        geoId,
                        req.langdetect.id,
                    );
                })
                .then(linguistics => {
                    region.title = linguistics.nominative;

                    return region;
                });
        })
        .catch(logError('getRegionByRequest', req));
}

// Получает список геоидентификаторов населенных пункта:
// - текущий населенный пункт,
// - райцентр,
// - столица региона,
// - стлица федерального округа,
// - столица страны.
//
// Пример для села Антипино: [117180, 100845, 55, 54, 213]
// - Антипино - 117180,
// - Нижняя Тавда (райцентр) - 100845,
// - Тюмень (столица Тюменской области) - 55,
// - Екатеринбург (столица Уральского Федерального Округа) - 54,
// - Москва - 213.
export function getCityHierarchy(geoId) {
    const geoIds = [];

    return geobase
        .parents(geoId)
        .then(regionIds =>
            Promise.all(
                regionIds.map(regionId =>
                    geobase.regionById(regionId).then(region => {
                        // Типы регионов:
                        // 3 - страна
                        // 4 - федеральный округ
                        // 5 - субъект федерации (Свердловская область, Башкирия)
                        // 6 - город
                        // 7 - село
                        // 10 - район субъекта федерации (Новолялинский район Свердловской области)
                        if (region.type === 6 || region.type === 7) {
                            geoIds.push(regionId);
                        } else if (
                            region.type === 10 ||
                            region.type === 5 ||
                            region.type === 4 ||
                            region.type === 3
                        ) {
                            if (region.chief_region) {
                                geoIds.push(region.chief_region);
                            } else if (!geobase.__syncGeobase__) {
                                return geobase.chiefRegionId(regionId).then(
                                    chiefRegionId => {
                                        geoIds.push(chiefRegionId);
                                    },
                                    // eslint-disable-next-line @typescript-eslint/no-empty-function
                                    () => {},
                                );
                            }
                        }
                    }),
                ),
            ),
        )
        .then(() => geoIds)
        .catch(logError('getCityHierarchy'));
}
