import got, {Response} from 'got';
import {FORMAT_HTTP_HEADERS} from 'opentracing';

import GotOptions from '../../common/interfaces/server/GotOptions';
import IExpressRequest from '../../common/interfaces/IExpressRequest';

import hasValue from '../../common/lib/url/hasValue';
import getTracer from './jaeger/getTracer';
import addErrorInfoToTracerSpan from './jaeger/addErrorInfoToTracerSpan';
import getFullUriByGotOptions from './getFullUriByGotOptions';

const xRealIpHeader = 'X-Rasp-Real-IP';
const xYaUaasExperimentsHeader = 'X-Ya-Uaas-Experiments';

export interface IApiTracerOptions {
    operationName: string;
}

/**
 * Обертка над got для запросов к api
 * @param {Object} req - объект запроса express
 * @param {Object} tracerOptions - объект опций для трассировки
 * @return {Function}
 */
export default function getApiGot(
    req: IExpressRequest,
    tracerOptions: IApiTracerOptions,
): (options: GotOptions) => Promise<Response<any>> {
    if (!req) {
        throw new Error('Parameter "req" is required');
    }

    return options => {
        // Создаем дочерний span трэйсинга
        const operationName = tracerOptions?.operationName;
        const tracer = getTracer();
        const childSpan = tracer.startSpan(
            `${options.hostname}${operationName || options.path}`,
            {childOf: req.rootSpan},
        );

        // прокидываем x-real-ip
        const realIp = req.headers['x-real-ip'];
        const {headers = {}, query = {}} = options;

        if (realIp && !headers[xRealIpHeader]) {
            headers[xRealIpHeader] = realIp;
        }

        if (req.flags) {
            headers[xYaUaasExperimentsHeader] = JSON.stringify(req.flags);
        }

        tracer.inject(childSpan, FORMAT_HTTP_HEADERS, headers);

        if (req.requestId && !query._rid) {
            query._rid = req.requestId;
        }

        // чистим query от неопределенных параметров
        Object.entries(query).forEach(([key, value]) => {
            if (!hasValue(value)) {
                delete query[key];
            }

            if (typeof value === 'boolean') {
                query[key] = value ? 'true' : 'false';
            }
        });

        const gotOptions = {
            ...options,
            headers,
            query,
        };

        childSpan.addTags({
            query,
            headers,
            uri: getFullUriByGotOptions(gotOptions),
        });

        return got({}, gotOptions)
            .catch(err => {
                addErrorInfoToTracerSpan(childSpan, err);

                throw err;
            })
            .finally(() => {
                childSpan.finish();
            });
    };
}
