import {Request, Response} from 'express';

import IExpressRequest from '../../common/interfaces/IExpressRequest';

import cookieSetter from '../helpers/cookies';

const FLAGS_PREFIX = '_exp_';

function parseValue(value: string): string | boolean {
    switch (value) {
        case '1':
        case 'true':
            return true;
        case '0':
        case 'false':
            return false;

        default:
            return value;
    }
}

export function getUserFlags(
    req: Request & Pick<IExpressRequest, 'networkTraits'>,
    res: Response,
): Record<string, string | boolean> {
    const userFlags = {};
    const cookies = cookieSetter(req, res);

    // Считываем сохранённые значения из кук
    Object.entries(req.cookies).forEach(([key, value]) => {
        if (key.startsWith(FLAGS_PREFIX) && typeof value === 'string') {
            userFlags[key.substr(FLAGS_PREFIX.length)] = parseValue(value);
        }
    });

    // Считываем новые значения из параметров запроса
    Object.entries(req.query).forEach(([key, value]) => {
        if (key.startsWith(FLAGS_PREFIX) && typeof value === 'string') {
            const flagsKey = key.substr(FLAGS_PREFIX.length);

            // Если зачение задано - добавляем к флагам и сохраняем в куку
            // Если значение пустое - убираем флаг, сносим куку
            if (value) {
                userFlags[flagsKey] = parseValue(value);
                cookies.set(key, value, {
                    expires: new Date(Date.now() + 3600000 * 24), // сутки
                });
            } else if (cookies.get(key)) {
                cookies.expire(key);
                delete userFlags[flagsKey];
            }
        }
    });

    return userFlags;
}
