import fs from 'fs';
import path from 'path';
import yaConfig from '@yandex-int/yandex-config';
import {momentTimezone as moment} from '../../reexports';

import {ALL_TYPE} from '../../common/lib/transportType';
import {getGeoIdByRequest} from '../helpers/geobaseHelper';

const {logPaths} = yaConfig();

function ensureDirectory(filename) {
    const dirname = path.dirname(filename);

    try {
        fs.statSync(dirname);
    } catch (err) {
        fs.mkdirSync(dirname);
    }
}

function buildTskvRow(entry) {
    const keyValues = Object.keys(entry)
        .sort()
        .map(key => {
            let value = entry[key];

            if (typeof value === 'object') {
                value = JSON.stringify(value);
            }

            return `${key}=${value}`;
        });
    const tskv = keyValues.join('\t');

    return `tskv\ttskv_format=rasp-users-search-log\t${tskv}\n`;
}

function logToFile(filename, record, useJson) {
    const now = moment.utc();

    ensureDirectory(filename);
    const logEntry = {
        ...record,

        service: 'rasp',
        unixtime: now.unix(),
        eventtime: now.format('YYYYMMDDHHmmss'),
    };

    fs.appendFileSync(
        filename,
        useJson ? `${JSON.stringify(logEntry)}\n` : buildTskvRow(logEntry),
    );
}

function logReqToFile(filename, req, record) {
    getGeoIdByRequest(req).then(geoid => {
        logToFile(filename, {
            ...record,

            yandexuid: req.cookies.yandexuid,
            passportuid: req.blackbox?.uid,
            userip: req.ip,
            geoid,
            referer: req.get('referer'),
            request_id: req.requestId,
        });
    });
}

function countTransportSegments(segments) {
    return segments.reduce((result, {transport}) => {
        result[transport.code] = (result[transport.code] || 0) + 1;

        return result;
    }, {});
}

function recordFromQuery(query) {
    return {
        from_id: query.pointFrom,
        national_version: query.nationalVersion,
        to_id: query.pointTo,
        transport_type: query.transportType || ALL_TYPE,
        when: query.when,
    };
}

function extendSearchRecord(record) {
    return {
        ...record,

        // для совместимости с данными авиабилетов
        return_date: null,
        klass: null,
        adults: null,
        children: null,
        infants: null,
    };
}

export function logSearchResult({
    req,
    query,
    result,
    logPathsConfig = logPaths,
}) {
    const record = recordFromQuery(query);

    record.t_type_counts = countTransportSegments(result.segments);
    logReqToFile(logPathsConfig.search, req, extendSearchRecord(record));
}

export function logSearchTransfersResult({
    req,
    query,
    result,
    logPathsConfig = logPaths,
}) {
    const record = recordFromQuery(query);

    record.transfer_count = result
        ? result.map(({segments}) => countTransportSegments(segments))
        : null;
    logReqToFile(logPathsConfig.search, req, extendSearchRecord(record));
}

export function logBusTariffs({query, data, logPathsConfig = logPaths}) {
    logToFile(
        logPathsConfig.yBus,
        {
            ...recordFromQuery(query),
            ...data,
        },
        true,
    );
}
