import v8 from 'v8';

import formatBytes from '../../../common/lib/format/formatBytes';

interface IHeapSpaceStatistic {
    space_size: string;
    space_used_size: string;
    space_available_size: string;
    physical_space_size: string;
}

type IResult = Record<string, IHeapSpaceStatistic>;

export default function getFormattedHeapSpaceStatistics(): IResult {
    const stats = v8.getHeapSpaceStatistics();

    const resStat = stats.reduce(
        (result, stat) => {
            result.space_size += stat.space_size;
            result.space_used_size += stat.space_used_size;
            result.space_available_size += stat.space_available_size;
            result.physical_space_size += stat.physical_space_size;

            return result;
        },
        {
            space_name: 'result',
            space_size: 0,
            space_used_size: 0,
            space_available_size: 0,
            physical_space_size: 0,
        },
    );

    const formattedStats = stats.concat(resStat).map(stat => {
        return {
            space_name: stat.space_name,
            space_size: formatBytes(stat.space_size),
            space_used_size: formatBytes(stat.space_used_size),
            space_available_size: formatBytes(stat.space_available_size),
            physical_space_size: formatBytes(stat.physical_space_size),
        };
    });

    return formattedStats.reduce<IResult>((result, {space_name, ...rest}) => {
        result[space_name] = rest;

        return result;
    }, {});
}
