import got from 'got';

import getApiGot from './getApiGot';
import logger from '../logger';
import getFullUriByGotOptions from './getFullUriByGotOptions';

const pathToFile = 'server/helpers/timedGot';

function getErrorInfo(err, gotOptions) {
    return `url: ${getFullUriByGotOptions(gotOptions)}, error: ${err.message}`;
}

function retryBuilder(retries, gotOptions) {
    return function (iteration, err) {
        if (iteration > retries) {
            return 0;
        }

        logger.info(
            pathToFile,
            `Retry query to API. ${getErrorInfo(err, gotOptions)}`,
        );

        return 1000 * Math.pow(2, iteration) + Math.random() * 100;
    };
}

function timedWrapper(gotOptions, gotFunction = got) {
    const startTime = Date.now();
    const {path, retries} = gotOptions;

    return gotFunction({
        ...gotOptions,
        retries: retryBuilder(retries !== undefined ? retries : 2, gotOptions),
    }).then(res => {
        const time = (Date.now() - startTime) / 1000;
        const message = `path: ${path}, time: ${time} s, uri: ${getFullUriByGotOptions(
            gotOptions,
        )}`;

        logger.http(pathToFile, message);

        return res;
    });
}

export default function timedGot(gotOptions) {
    return timedWrapper(gotOptions);
}

export function apiTimedGot(gotOptions, req, tracerOptions) {
    return timedWrapper(gotOptions, getApiGot(req, tracerOptions));
}
