const path = require('path');
const vowFs = require('vow-fs');

/**
 * Метод формирования контента входной точки для текстов
 * @param {string[]} languages - все доступные языки
 * @param {string} defaultLanguage - язык используемый по умолчания
 * @return {string}
 */
function getEntryPointContent(languages, defaultLanguage) {
    if (languages.length === 0) {
        throw new Error('no locales to export');
    }

    const languageImports = languages
        .filter(language => language !== defaultLanguage)
        .map(language => {
            return (
                `if (process.env.BUNDLE_LANGUAGE === "${language}") {\n` +
                `    exportModule = require("./${language}.js").default;\n` +
                '}'
            );
        })
        .join(' else ');

    const defaultImport =
        ' else {\n' +
        `    exportModule = require("./${defaultLanguage}.js").default;\n` +
        '}';

    return (
        "import Keyset from '../../interfaces/Keyset';\n\n" +
        'let exportModule: Keyset;\n\n' +
        `${languageImports}${defaultImport}\n\n` +
        'export default exportModule;\n'
    );
}

/**
 * Метод для создания файла - входной точки, которая объединяет все локали
 * @param {string} keysetPath
 * @param {string[]} languages - все доступные языки
 * @param {string} defaultLanguage - язык используемый по умолчания
 * @return {Promise}
 */
function createEntryPoint(keysetPath, languages, defaultLanguage) {
    return vowFs
        .remove(path.join(keysetPath, 'index.js'))
        .catch(() => true)
        .then(() =>
            vowFs.write(
                path.join(keysetPath, 'index.ts'),
                getEntryPointContent(languages, defaultLanguage),
                'utf8',
            ),
        );
}

module.exports = {
    getEntryPointContent,
    createEntryPoint,
};
