/* eslint-disable no-console */
require('../../transpilerOnFly');

const path = require('path');
const fs = require('fs');
const vowFs = require('vow-fs');
const Vow = require('vow');
const tanker = require('./tanker');
const config = require('../configs/default');
const argv = require('yargs').argv;

// Библиотеки y-i18n-layer и compact-tl - недокументированная магия.
const yI18NLayer = require('enb-y-i18n/lib/y-i18n-layer');
const CompactTL = require('compact-tl').CompactTL;
const createEntryPoint = require('./bundling').createEntryPoint;

const compactTl = new CompactTL();

compactTl.use(yI18NLayer.create({pluralFunctionName: 'plural'}));

const NBSP_REGEXP = /&nbsp;/gi;
const CHAR_NBSP = '\u00a0';

const keysetsFolder = path.join(__dirname, '../../common/i18n');

// формируем список текущих кейсетов
const existsKeysets = [];

fs.readdirSync(keysetsFolder).forEach(file => {
    existsKeysets.push(file);
});

const keysetIsValid = function (keyset) {
    const keysetData = keyset.data;

    if (keysetData) {
        return Object.keys(keysetData).reduce(
            (allLanguagesAreValid, language) => {
                const langKeyset = keysetData[language];

                return (
                    allLanguagesAreValid &&
                    Object.keys(langKeyset).reduce((languageIsValid, key) => {
                        return (
                            languageIsValid &&
                            Boolean(langKeyset[key] && langKeyset[key].length)
                        );
                    }, true)
                );
            },
            true,
        );
    }

    return false;
};

const download = function (outputPath, tankerParams) {
    console.log('Tanker Download started.');
    const tankerObj = tanker.Tanker(tankerParams);

    return Vow.when(tankerObj.getPrjKeysets())
        .then(keysetInfos => {
            return Vow.all(
                keysetInfos
                    .filter(keysetInfo => {
                        // скачиваем только тот кейсет, что указан в параметре "name"
                        // или все, если параметр не указан
                        // или обновляем только имеющиеся кейсеты при наличии
                        // параметра --updateExists
                        return (
                            (!argv.name && !argv.updateExists) ||
                            (argv.name && argv.name === keysetInfo.name) ||
                            (argv.updateExists &&
                                existsKeysets.indexOf(keysetInfo.name) !== -1)
                        );
                    })
                    .map(keysetInfo => {
                        const opts = {safe: 'yes'};

                        return tankerObj
                            .getPrjKeyset(keysetInfo.name, opts)
                            .then(keysetData => {
                                return {
                                    name: keysetInfo.name,
                                    data: JSON.parse(keysetData),
                                };
                            });
                    }),
            );
        })
        .then(res => {
            return Vow.all(
                res.filter(keysetIsValid).map(keyset => {
                    if (!outputPath) {
                        throw new Error(
                            'Usage error: outputPath parameter is not defined.',
                        );
                    }

                    const destPath = path.join(outputPath, keyset.name);

                    return _storeKeyset(keyset, destPath);
                }),
            );
        })
        .then(() => {
            console.log('Tanker download complete.');
        })
        .catch(e => {
            console.error('Tanker Error', e);
        });
};

function _storeKeyset(keyset, keysetPath) {
    const keysetData = keyset.data;
    const languages = Object.keys(keysetData);

    return vowFs.makeDir(keysetPath).then(() => {
        return Vow.all(
            Object.keys(keysetData).map(language => {
                const filePath = path.join(keysetPath, `${language}.js`);
                const langKeyset = keysetData[language];
                let keysetResult = `    name: ${JSON.stringify(
                    keyset.name,
                )},\n    data: {\n`;

                Object.keys(langKeyset).forEach((key, i, arr) => {
                    keysetResult += `        ${JSON.stringify(key)}: ${compactTl
                        .process(langKeyset[key])
                        .replace(NBSP_REGEXP, CHAR_NBSP)}${
                        i === arr.length - 1 ? '' : ','
                    }\n`;
                });
                keysetResult += '    }\n';

                const content =
                    'import plural from "../_utils/plural";\n' +
                    'import exporter from "../_utils/exporter";\n\n' +
                    `export default exporter({\n${keysetResult}});\n`;

                return vowFs.exists(filePath).then(exists => {
                    if (exists) {
                        return vowFs.read(filePath, 'utf8').then(oldContent => {
                            if (oldContent !== content) {
                                console.log(`${filePath} updated`);

                                return vowFs.write(filePath, content, 'utf8');
                            }

                            // console.log(filePath + ': nochanges');
                            return null;
                        });
                    }

                    console.log(`${filePath} created`);

                    return vowFs.write(filePath, content, 'utf8');
                });
            }),
        ).then(() => createEntryPoint(keysetPath, languages, 'ru'));
    });
}

if (!argv.name && !argv.updateExists) {
    console.error(
        'Необходимо указать параметр --name: npm run i18n.get -- --name=foo' +
            ' или параметр --updateExists для обновления' +
            ' всех уже имеющихся кейсетов: npm run i18n.get -- --updateExists',
    );
    process.exit(1);
}

download(keysetsFolder, config.tanker);
