const https = require('https');
const qs = require('querystring');
const Vow = require('vow');
const extend = require('extend');

/**
 * @param {Object} options - набор параметров из config настроек
 * @constructor
 */
function Tanker(options) {
    // prj, host, port, rev, token, langs
    this._init(options);

    return this;
}

Tanker.prototype = {
    /**
     * @param {Object} params Параметры инициализации
     * @param {string} params.host
     * @param {number} params.port
     * @param {string} params.token
     * @param {string} params.prj
     * @param {string} params.rev
     */
    _init(params) {
        if (params) {
            ['host', 'port', 'token', 'prj', 'rev'].forEach(p => {
                if (params.hasOwnProperty(p)) {
                    this[`_${p}`] = params[p];
                }
            });
        }
    },

    /**
     * @private
     * @param {string} path_
     * @param {Object} [reqOptions]
     * @param {Object} [data]
     * @return {Promise}
     */
    _doRequest(path_, reqOptions, data) {
        const opt = this.apiParams(reqOptions);
        let respData = '';

        opt.path = path_;

        if (data) {
            data = qs.stringify(data);
            opt.headers['Content-Length'] = data.length;
        }

        const deferred = Vow.defer();
        /** {https.ClientRequest} */
        const req = https.request(opt, res => {
            res.setEncoding('utf8');
            res.on('data', chunk => {
                respData += chunk;
            }).on('end', () => {
                if (res.statusCode === 200) {
                    deferred.resolve(respData);
                } else {
                    deferred.reject(
                        new Error(`${respData}\nDEBUG: ${JSON.stringify(opt)}`),
                    );
                }
            });
        });

        req.on('error', e => {
            deferred.reject(
                new Error(`${e.message}\nDEBUG: ${JSON.stringify(opt)}`),
            );
        });

        if (data) {
            req.write(data);
        }

        req.end();

        return deferred.promise();
    },

    /**
     * @private
     * @param {Object} [params]
     * @param {string} [params.keysetRU-id] Название кейсета
     * @param {string} [params.rev] Имя ветви, или идентификатор ревизии проекта
     * @param {string} [params.lang] Язык, который следует выгрузить
     * @param {boolean} [params.status=false] Используется как "status=unapproved"
     *      для выгрузки всех переводов в т.ч. неподтвержденных
     * @param {boolean} [params.safe=false] Выгражать ключи по умолчанию,
     *      вместо пустых переводов
     * @param {boolean} [params.noComments=false] Не выгружать комментарии
     * @return {Object} Нормализованные параметры
     */
    _prepareExportParams(params = {}) {
        const query = {
            'project-id': this._prj,
            'branch-id': params.rev || this._rev,
            language: false,
            status: params.status || false,
            safe: params.safe || false,
            'no-comments': params.noComments || false,
            'flat-keyset': 1,
        };

        if (params.langs) {
            query.language = params.langs;
            delete params.langs;
        }

        extend(query, params);

        for (const k in query) {
            if (query.hasOwnProperty(k) && !query[k]) {
                delete query[k];
            }
        }

        return query;
    },

    /**
     * Устанавливает/возвращает параметры вызова API
     * @param {Object} params
     * @return {Object}
     */
    apiParams(params) {
        const headers = {};

        if (this._token) {
            headers.Authorization = `OAuth ${this._token}`;
        }

        const options = {
            host: this._host,
            port: this._port,
            headers,
            method: 'GET',
        };

        if (typeof params === 'object') {
            extend(options, params);
        }

        if (options.method === 'POST') {
            extend(options.headers, {
                'Content-Type': 'application/x-www-form-urlencoded',
            });
        }

        return options;
    },

    /**
     * @param {string} [branch]
     * @return {Array}
     */
    getPrjKeysets(branch) {
        if (!branch) {
            branch = this._rev;
        }

        return this._doRequest(
            `/admin/project/${this._prj}/keysets/?branch=${branch}`,
        ).then(data => {
            return JSON.parse(data).data.items;
        });
    },

    /**
     * Выгрузка кейсета `keysetId`
     *
     * @param {string} keysetId
     * @param {Object} [params]
     * @return {string} raw-данные
     */
    getPrjKeyset(keysetId, params) {
        const defaultParams = {
            'keyset-id': keysetId,
        };

        extend(defaultParams, params);

        const query = this._prepareExportParams(defaultParams);
        const format = defaultParams.format || 'json';

        return this._doRequest(
            `/keysets/${format}/?${qs.stringify(query)}`,
        ).then(data => {
            return data;
        });
    },
};

exports.Tanker = function (params) {
    return new Tanker(params);
};
