'use strict';

require('../defineEnvironment');

const environment = require('@yandex-int/yandex-environment');

if (environment === 'development') {
    require('../transpilerOnFly');
}

const path = require('path');
const recluster = require('recluster');
const mkdirp = require('mkdirp');
const FS = require('fs');
const log = require('./logger').default;

const config = require('@yandex-int/yandex-config')();
const prefix = `index.js (${config.app.version})`;

const moment = require('../reexports').momentTimezone;

moment.tz.setDefault(config.defaultTimezone);

const cluster = recluster(path.join(__dirname, 'worker.js'), {
    workers: config.cluster.workersCount,
});

cluster.on('exit', worker => {
    if (worker.process.exitCode) {
        log.error(
            prefix,
            `[${worker.process.pid}] Worker died (exit code: ${worker.process.exitCode})`,
        );
    }
});
cluster.on('disconnect', worker => {
    log.info(prefix, `[${worker.process.pid}] Worker disconnected`);
});

(function cleanupSocket() {
    const socket = config.server.socket;

    if (socket) {
        mkdirp.sync(path.dirname(socket));

        if (FS.existsSync(socket)) {
            FS.unlinkSync(socket);
        }
    }
})();

cluster.run();

process.on('SIGUSR2', () => {
    log.info(prefix, '[master] Got SIGUSR2, reloading cluster...');
    cluster.reload();
});

log.info(
    prefix,
    `[master] Spawned cluster, kill -s SIGUSR2 ${process.pid} to reload`,
);

if (environment === 'development') {
    require('watchr').watch({
        paths: [__dirname],
        ignorePaths: [`${__dirname}/node_modules`],
        listener(changeType, filePath) {
            if (filePath.match(/\.js$/)) {
                log.info(
                    prefix,
                    `[master] File ${filePath} has changed, reloading`,
                );
                cluster.reload();
            }
        },
    });
}
