import {Request, Response, NextFunction} from 'express';
import {FORMAT_HTTP_HEADERS} from 'opentracing';
import {v4 as uuid} from 'uuid';

import IExpressRequest from '../../common/interfaces/IExpressRequest';

import getTracer from '../helpers/jaeger/getTracer';
import addErrorInfoToTracerSpan from '../helpers/jaeger/addErrorInfoToTracerSpan';
import checkIsYandexNetwork from '../helpers/checkIsYandexNetwork';

const HEADER_JAEGER_DEBUG_ID = 'jaeger-debug-id';
const START_JAEGER_OPERATION_NAME = 'start-operation';

export default function addTracerSpan(
    req: Request &
        Pick<IExpressRequest, 'rootSpan' | 'networkTraits' | 'jaegerDebugId'>,
    res: Response,
    next: NextFunction,
): void {
    const jaegerDebugIdFromHeaders = req.headers[HEADER_JAEGER_DEBUG_ID];

    if (
        typeof jaegerDebugIdFromHeaders === 'string' &&
        jaegerDebugIdFromHeaders
    ) {
        req.jaegerDebugId = jaegerDebugIdFromHeaders;
    }

    if (!req.jaegerDebugId && checkIsYandexNetwork(req)) {
        req.jaegerDebugId = uuid();
        req.headers[HEADER_JAEGER_DEBUG_ID] = req.jaegerDebugId;
    }

    const tracer = getTracer();
    const parentContext = tracer.extract(FORMAT_HTTP_HEADERS, req.headers);
    const rootSpan = tracer.startSpan(START_JAEGER_OPERATION_NAME, {
        childOf: parentContext || undefined,
    });

    rootSpan.addTags({
        path: req.path,
        query: req.query,
        url: `${req.protocol}://${req.hostname}${req.url}`,
    });

    req.rootSpan = rootSpan;

    res.once('finish', () => {
        req.rootSpan.finish();
    });

    /** @see https://github.com/opentracing-contrib/javascript-express/issues/16 */
    res.once('abort', () => {
        req.rootSpan.finish();
    });

    res.once('error', err => {
        addErrorInfoToTracerSpan(req.rootSpan, err);
        req.rootSpan.finish();
    });

    next();
}
