import {NextFunction, Response, Request} from 'express';

import IExpressRequest from '../../common/interfaces/IExpressRequest';

import {getTouchAlternateLink} from '../../common/lib/url/altLinks';

// Регулярка для tunneller (https://wiki.yandex-team.ru/users/unikoid/tunneler/quickstart/)
const regExpIsTunnel = /ldev\.yandex-team\.ru/i;

// Регулярка regExpIsInfoPath нужна для исключения цикличесвого редиректа.
// Если path с ней совпадает, то ненужно отдавать редирект на тачевый домен для мобильных устройств
// Нюансы:
// 1. Страницы типа /info/convid-19, /info/convid-19-ua отдаются с нового стека, а остальные страницы /info/ со старого
// 2. На старом таче есть редиректы со страниц https://rasp.yandex.ru/info* на https://t.rasp.yandex.ru
const regExpIsInfoPath = /^\/info(?!\/covid-19)/;

function redirectToTouch(
    req: Request &
        Pick<IExpressRequest, 'isTouchDomain' | 'uatraits' | 'fullUrl'>,
    res: Response,
    backendAltLinks,
): boolean {
    if (
        req.isTouchDomain ||
        regExpIsTunnel.test(req.fullUrl) ||
        regExpIsInfoPath.test(req.path)
    ) {
        return false;
    }

    if (req.uatraits.isMobile && backendAltLinks.touch) {
        res.redirect(backendAltLinks.touch.href);

        return true;
    }

    return false;
}

export default function altRedirects(
    req: Request &
        Pick<
            IExpressRequest,
            | 'isTouchDomain'
            | 'uatraits'
            | 'fullUrl'
            | 'backendAltLinks'
            | 'altLinks'
        >,
    res: Response,
    next: NextFunction,
): void {
    const backendAltLinks = {
        touch: getTouchAlternateLink(req.fullUrl),
    };

    if (!redirectToTouch(req, res, backendAltLinks)) {
        req.backendAltLinks = backendAltLinks;
        req.altLinks = (req.altLinks || []).concat(
            Object.keys(backendAltLinks).map(
                service => backendAltLinks[service],
            ),
        );

        next();
    }
}
