import pick from 'lodash/pick';
import mapKeys from 'lodash/mapKeys';
import camelCase from 'lodash/camelCase';

import {MOBILE} from '../../common/lib/platforms';

import {setSeoQueryParams} from '../../common/actions/seoQueryParams';

const SEO_KEYS = [
    'req_id',
    'utm_source',
    'utm_medium',
    'utm_campaign',
    'utm_term',
    'utm_content',
    'from',
];

const SEO_CAMEL_CASE_KEYS = SEO_KEYS.map(camelCase);

/**
 * Функция возвращает query параметры необходимые для seo в camelCase
 *
 * @param {Object} query - query запроса
 * @return {Object}
 */
function getCamelCasedSeoQuery(query) {
    const seoQuery = pick(query, SEO_KEYS);

    return mapKeys(seoQuery, (val, key) => camelCase(key));
}

/**
 * Проверяет, надо ли записывать/перезаписывать seo куку
 *
 * @param {Object} seoQuery - seo поля из query запроса
 * @param {Object} oldSeoQueryCookie - предыдущее значение поля seoQuery из cookie
 * @return {boolean}
 */
function needToOverwriteSeoCookie(seoQuery, oldSeoQueryCookie) {
    return Object.entries(seoQuery).some(
        ([queryKey, queryVal]) => oldSeoQueryCookie[queryKey] !== queryVal,
    );
}

/**
 * Мидлавара записывает seo параметры из url в store
 *
 * @param {Function} next
 * @param {Object} cookies
 * @param {Object} req
 * @param {Object} store
 */
export default function storeSeoQueryParams({next, cookies, req, store}) {
    const {getState, dispatch} = store;
    const {platform} = getState();
    const {query} = req;

    const seoQueryCookie = pick(cookies.get('seoQuery'), SEO_CAMEL_CASE_KEYS);
    const seoQuery = getCamelCasedSeoQuery(query);

    let seoQueryToSet = {
        ...seoQueryCookie,
        device: platform === MOBILE ? 'touch' : 'desktop',
    };

    if (needToOverwriteSeoCookie(seoQuery, seoQueryCookie)) {
        seoQueryToSet = {
            ...seoQuery,
            device: seoQueryToSet.device,
        };

        cookies.set('seoQuery', seoQueryToSet);
    }

    dispatch(setSeoQueryParams(seoQueryToSet));

    next();
}
