/* eslint no-console: 0 */
import fs from 'fs';
import path from 'path';
import serialize from 'serialize-javascript';

import Platform from '../../common/interfaces/Platform';
import Environment from '../../common/interfaces/Environment';
import IStateFlags from '../../common/interfaces/state/flags/IStateFlags';
import IErrorCounterConfig from 'common/interfaces/IErrorCounterConfig';

interface IErrorCounterScriptParams {
    env: Environment;
    platform: Platform;
    experiments: IStateFlags;
}

interface IInlineErrorCounterScriptParams extends IErrorCounterScriptParams {
    nonce: string;
}

const LIB_PATH = path.join(
    __dirname,
    '../../',
    'node_modules',
    '@yandex-int',
    'error-counter',
    'dist',
);

const errorCounterScripts = [
    'interfaceWithoutRum.min.js',
    'implementation.min.js',
    'filters.min.js',
    'sendWithoutRum.min.js',
    'logError.min.js',
    'logAjaxError.min.js',
    'logEvent.min.js',
    'traceUnhandledRejection.min.js',
].map(fileName => path.join(LIB_PATH, fileName));

const errorCounterScriptsContent = errorCounterScripts
    .map((scriptPath: string) => {
        const fileContent = fs.readFileSync(scriptPath);

        return fileContent.toString();
    })
    .join('\n');

const getErrorCounterScriptContent = (
    params: IErrorCounterScriptParams,
    config: IErrorCounterConfig,
): string => {
    const {env, platform, experiments} = params;

    const enabledExperiments = Object.entries(experiments)
        .filter(([, expValue]) => expValue !== false && expValue !== undefined)
        .map(([expKey, expValue]) => `${expKey}=${expValue}`);

    const errorCounterSettings = {
        ...config,
        env,
        platform: platform === Platform.mobile ? 'touch' : 'desktop',
        experiments: enabledExperiments,
    };

    return `${errorCounterScriptsContent}
        Ya.Rum.initErrors(${serialize(errorCounterSettings)})`;
};

const errorCounterScript = (
    params: IInlineErrorCounterScriptParams,
    config: IErrorCounterConfig,
): string => {
    const {nonce, ...rest} = params;

    return `<script nonce="${nonce}">${getErrorCounterScriptContent(
        rest,
        config,
    )}</script>`;
};

export default errorCounterScript;
