import {THREAD_PAGE_NAME} from '../../common/routes/thread';
import {TRANSPORT_CITY_PAGE_NAME} from '../../common/routes/transportCity';
import {TRANSPORT_PAGE_NAME} from '../../common/routes/transport';
import {STATION_PAGE_NAME} from '../../common/routes/station';

import {Manifest, Bundle} from 'react-loadable/webpack';

// eslint-disable-next-line no-duplicate-imports
import {getBundles} from 'react-loadable/webpack';

/*
Функция определяет список css-файлов необходимых для конкретной страницы.

Эта функции родилась в результате некоторой несостыковки между серверным рендерингом в виде потока и
использованием модуля react-loadable.
Дело в том, что полный список необходимых css-файлов мы узнаем толко после полного рендера приложения,
а вот тег <head> со стилями мы отдаем еще до начала рендеринга. Если не предпринимать никаких дополнительный
действий, то в случае загрузки страницы, выделенной в отдельный чанк (например ThreadPage), клиент сначала получит страницу без стилей,
и выглядеть она будет поломанной. Затем отработает js, react-loadable подсунет в <head> недостающий css-файл,
он загрузится, и страница отобразится в нормальном виде. Суть этой функции как раз состояит в том, чтобы избежать этого мигания.

С js-файлами такой проблемы нет, потому что мы  подключем их в конце документа.
*/
export default function (
    reactLoadableJson: Manifest,
    pageType: string,
    loadedCssChunks: Bundle[] = [],
): Bundle[] {
    const loadedCssFiles = loadedCssChunks.map(chunk => chunk.file);
    const moduleId = getModuleIdByPageType(pageType);

    if (!moduleId) {
        return [];
    }

    return getBundles(reactLoadableJson, [moduleId]).filter(
        ({file}) => file.endsWith('.css') && !loadedCssFiles.includes(file),
    );
}

function getModuleIdByPageType(pageType: string): string | undefined {
    switch (pageType) {
        case THREAD_PAGE_NAME:
            return '../ThreadPage';
        case TRANSPORT_CITY_PAGE_NAME:
            return '../TransportCityPage';
        case TRANSPORT_PAGE_NAME:
            return '../TransportPage';
        case STATION_PAGE_NAME:
            return '../StationPage';
    }

    return undefined;
}
