import {React} from '../../common/components/base';

import yaConfig from '@yandex-int/yandex-config';
import environment from '@yandex-int/yandex-environment';
import {getBundles} from 'react-loadable/webpack';

import {
    YANDEX_SANS_CLASS,
    YANDEX_SANS_COOKIE,
} from '../../client/helpers/yandexSansDownloadObserver';

import IStore from '../../common/interfaces/IStore';
import IExpressRequest from '../../common/interfaces/IExpressRequest';
import IExpressResponse from '../../common/interfaces/IExpressResponse';

import getReactBundle from '../helpers/getReactBundle';
import getCrowdTesting from '../helpers/getCrowdTesting';
import {getFaviconsTags} from '../helpers/favicons';
import metrika from '../helpers/counters/metrika';
import {getHostForBundle} from '../../webpack/staticUtils';
import getLoadableJson from './getLoadableJson';
import getSvgSprite from './getSvgSprite';
import errorRenderingScript from './errorRenderingScript';
import getCssLoadableChunksByPageType from './getCssLoadableChunksByPageType';
import addCssChunksScript from './addCssChunksScript';
import errorCounterScript from './errorCounterScript';

import generateTagsAsString from '../../common/lib/meta/generateTagsAsString';
import {encodeSpecialCharacters} from '../../common/lib/stringUtils';
import getMetaInformation from '../../common/lib/meta/metaTags';
import {getAdsScripts} from './getAdsScripts';

const config = yaConfig();

const stateReplacer = (key: string, value: any): any => {
    return typeof value === 'string' ? encodeSpecialCharacters(value) : value;
};

export default async function template(
    req: IExpressRequest,
    store: IStore,
    res: IExpressResponse,
): Promise<string> {
    if (req.aborted || res.finished) {
        return '';
    }

    // nonce - сгенерированный CSP код, необходимый для подписи inline-скриптов
    const {nonce, abtInfo} = req;
    const state = store.getState();
    const {
        flags,
        platform,
        language,
        isTouch,
        user,
        page: {current: pageType},
    } = state;
    const modules: string[] = [];
    // Заголовок для тестирования ошибок в компонентах
    const throwError = req.get('X-Rasp-Throw-Error') || '';
    const modernBrowser = user.browser.isModern;

    const isCrowdTesting = getCrowdTesting(req);

    if (typeof modernBrowser === 'undefined') {
        return Promise.reject(
            `Не удалось получить признак isModern для браузера. ${JSON.stringify(
                req.uatraits,
            )}`,
        );
    }

    const {
        Root,
        Helmet,
        Loadable,
        moment,
        ReactDOMServer,
        Provider,
        StaticContext,
    } = getReactBundle(platform, language, modernBrowser);

    moment.locale(language);

    const report = (moduleName: string): void => {
        modules.push(moduleName);
    };

    const errorCounterParams = {
        nonce,
        env: environment,
        platform,
        experiments: flags,
    };

    const staticContexValue = {
        isCrowdTesting,
    };

    const reactApp = (
        <StaticContext.Provider value={staticContexValue}>
            <Provider store={store}>
                <Loadable.Capture report={report}>
                    <Root throwError={throwError} />
                </Loadable.Capture>
            </Provider>
        </StaticContext.Provider>
    );

    // Собираем данные для рендера фавиконок на стороне сервера
    const favicons = getFaviconsTags();

    // SVG спрайт
    const spriteContent = getSvgSprite({platform, language, modernBrowser});
    const bundleHost = getHostForBundle({platform, language, modernBrowser});
    const loadableJson = getLoadableJson({platform, language, modernBrowser});

    await Loadable.preloadAll();

    return new Promise((resolve, reject) => {
        if (req.aborted || res.finished) {
            return resolve('');
        }

        const rootContentStream: NodeJS.ReadableStream & {
            destroy?: () => void;
        } = ReactDOMServer.renderToNodeStream(reactApp);

        const forceFinishStreams = (
            err: any = new Error('forceFinishStream'),
        ): void => {
            rootContentStream.unpipe(res);

            if (typeof rootContentStream.destroy === 'function') {
                rootContentStream.destroy();
            }

            if (!res.finished) {
                res.end();
            }

            reject(err);
        };

        const timeoutStream = setTimeout(() => {
            forceFinishStreams(new Error('timeout rootContentStream'));
        }, 20000);

        const onErrorStream = (err: any): void => {
            clearTimeout(timeoutStream);
            forceFinishStreams(err);
        };

        rootContentStream.once('error', onErrorStream);
        res.once('error', onErrorStream);

        res.once('finish', () => {
            resolve('');
        });

        // Так как react-helmet не умеет асинхронный рендеринг приходится рендерить Helmet
        // до рендеринга самого приложения, чтобы получить содержимое head и отдать его клиенту сразу
        Helmet.renderStatic();
        const HelmetForRender: any = Helmet; // typeScript ругается на тип react-helmet, так что пришлось закостылять

        ReactDOMServer.renderToString(
            <HelmetForRender {...getMetaInformation(state)} />,
        );
        const head = Helmet.renderStatic();

        // получаем css-чанки необходимые для конкретной страницы
        const cssChunks = getCssLoadableChunksByPageType(
            loadableJson,
            pageType || '',
        );

        const fontsLoadedClass = req.cookies[YANDEX_SANS_COOKIE]
            ? `class=${YANDEX_SANS_CLASS}`
            : '';

        res.setHeader('Content-Type', 'text/html; charset=utf-8');
        /* eslint-disable indent */
        res.write(
            `<!doctype html>
            <html ${fontsLoadedClass} ${head.htmlAttributes.toString()}>
                <head>
                    ${head.title.toString()}

                    ${generateTagsAsString('link', favicons.link)}
                    ${head.link.toString()}
                    ${head.script.toString()}

                    ${generateTagsAsString('meta', favicons.meta)}
                    ${head.meta.toString()}

                    <link rel="stylesheet" href="${bundleHost}/app.css" />
                    ${cssChunks
                        .map(
                            chunk =>
                                `<link rel="stylesheet" href="${bundleHost}/${chunk.file}" />`,
                        )
                        .join()}
                    ${errorCounterScript(
                        errorCounterParams,
                        config.errorCounter,
                    )}
                    ${getAdsScripts(nonce)}
                </head>
                <body>
                    ${errorRenderingScript(nonce)}
                    ${spriteContent}
                    <script type="text/javascript" nonce="${nonce}">
                        window.ENV = '${environment}';
                        window.STATIC_CONTEXT = ${JSON.stringify(
                            staticContexValue,
                            stateReplacer,
                        )};
                        window.INITIAL_STATE = ${JSON.stringify(
                            state,
                            stateReplacer,
                        )};
                        ${
                            req.mockParams
                                ? `window.MOCK = ${JSON.stringify({
                                      ...req.mockParams,
                                      now: req.mock ? req.mock.getNow() : '',
                                  })};`
                                : ''
                        };
                    </script>
                    <div id="root">`,
            'utf8',
        );

        rootContentStream.once('end', () => {
            Helmet.renderStatic();
            clearTimeout(timeoutStream);

            if (res.finished) {
                return;
            }

            const bundles = getBundles(loadableJson, modules);
            const jsChunks = bundles.filter(bundle =>
                bundle.file.endsWith('.js'),
            );
            // проверяем необходимость загрузки css-файлов
            const loadedCssFiles = cssChunks.map(({file}) => file);
            const finalCssChunks = bundles.filter(
                ({file}) =>
                    file.endsWith('.css') && !loadedCssFiles.includes(file),
            );

            res.write(
                `</div>
                    ${addCssChunksScript(nonce, finalCssChunks, bundleHost)}
                    <script type="text/javascript" src="${bundleHost}/libs.chunk.js"></script>
                    <script type="text/javascript" src="${bundleHost}/app.js"></script>
                    ${jsChunks
                        .map(
                            chunk =>
                                `<script src="${bundleHost}/${chunk.file}"></script>`,
                        )
                        .join('')}
                    ${metrika({nonce, isTouch, flags, abtInfo})}

                    <div style="display: none;" id="appFullLoaded"></div>
                </body>
            </html>`,
                'utf8',
            );

            res.end();
        });
        /* eslint-enable indent */

        rootContentStream.pipe(res, {end: false});
    });
}
