/* eslint-disable no-console */
import fsExtra from 'fs-extra';
import path from 'path';
import yaConfig from '@yandex-int/yandex-config';

import getAppVersion from './getAppVersion';
import exec from '../exec';
import execShellCommand from '../execShellCommand';
import getDockerName from './getDockerName';

const config = yaConfig();

const packageJsonPath = path.join(__dirname, '../../package.json');
const supervisorAppConfPath = path.join(
    __dirname,
    '../../docker/supervisor/conf.d/200-application.conf',
);
const dockerFilePath = path.join(__dirname, '../../Dockerfile');
const regexpDeleteForProd =
    /[=]==startProdDelete===(.|\n)+?===endProdDelete===/gm;

async function build(): Promise<void> {
    const appVersion = getAppVersion();
    const packageJsonData = await fsExtra.readFile(packageJsonPath, {
        encoding: 'utf8',
    });
    const supervisorAppConfData = await fsExtra.readFile(
        supervisorAppConfPath,
        {encoding: 'utf8'},
    );

    // Прописываем версию приложения
    await fsExtra.writeFile(
        packageJsonPath,
        packageJsonData.replace('DEB_VERSION', appVersion),
        {encoding: 'utf-8', flag: 'w'},
    );
    await fsExtra.writeFile(
        supervisorAppConfPath,
        supervisorAppConfData.replace('%%DEB_VERSION%%', appVersion),
        {encoding: 'utf-8', flag: 'w'},
    );

    // Формируем лог версии для страницы /version
    const {stdout: commitLog} = await execShellCommand(
        'arc log $(arc root)/travel/frontend/rasp -n 10',
        false,
    );
    const vesrionData = `App version: ${appVersion}\n\n${commitLog}`;

    await fsExtra.writeFile(config.pathToVersionFile, vesrionData, {
        encoding: 'utf8',
        flag: 'w',
    });

    // Патчим Dockerfile, чтобы продакшен-билд работал быстрее
    const dokerFileData = await fsExtra.readFile(dockerFilePath, {
        encoding: 'utf8',
    });

    await fsExtra.writeFile(
        dockerFilePath,
        dokerFileData.replace(regexpDeleteForProd, ''),
        {encoding: 'utf-8', flag: 'w'},
    );

    // Билдим контейнер
    const dockerTag = getDockerName(appVersion);

    await exec(`docker build --no-cache -t=${dockerTag} ./`);

    // Достаем из контейнера статику и выкладываем ее на s3
    const buildStaticPath = '/staticBuild';
    const awsBucket = 'rasp/s/morda-front';
    const awsUrl = 'https://s3.mds.yandex.net';

    await execShellCommand(
        `id=$(docker create ${dockerTag}) && docker cp $id:/app/static ${buildStaticPath} && docker rm -v $id`,
    );

    // На этом этапе должны существовать переменные окружения AWS_ACCESS_KEY_ID и AWS_SECRET_ACCESS_KEY
    await execShellCommand(
        `aws s3 --endpoint-url=${awsUrl} sync ${buildStaticPath}/_ s3://${awsBucket}/_/`,
    );
    await execShellCommand(`rm -rf ${buildStaticPath}/_`);
    await execShellCommand(
        `aws s3 --endpoint-url=${awsUrl} sync ${buildStaticPath} s3://${awsBucket}/${appVersion}/`,
    );

    await exec(`docker push ${dockerTag}`);
}

build()
    .then(() => {
        console.log('Билд прошел успешно');
        process.exit(0);
    })
    .catch(error => {
        console.error(error);
        console.error('Билд не удался');
        process.exit(1);
    });
