import IDataRequest from '../../../common/interfaces/tests/IDataRequest';
import Tld from '../../../common/interfaces/Tld';
import Lang from '../../../common/interfaces/Lang';
import StationDateSpecialValue from '../../../common/interfaces/date/StationDateSpecialValue';
import StationEventList from '../../../common/interfaces/state/station/StationEventList';
import IStateFlags from '../../../common/interfaces/state/flags/IStateFlags';
import StationSubtype from '../../../common/interfaces/state/station/StationSubtype';
import StationType from '../../../common/interfaces/state/station/StationType';
import StationTime from '../../../common/interfaces/state/station/StationTime';

import cartesianProduct from '../../../common/lib/cartesianProduct';
import {stationUrl} from '../../../common/lib/url/stationUrl';
import getAvailableLanguages from '../../../common/lib/lang/getAvailableLanguages';
import getMobileHost from './getMobileHost';
import getDesktopHost from './getDesktopHost';
import getAllTlds from './getAllTlds';

const allTlds = getAllTlds();
const allLanguages = Object.values(Lang);
const allSpecialDates = Object.values(StationDateSpecialValue);
const allEvents = Object.values(StationEventList);

interface IGetDataForRequestsAirports {
    idsAirports: number[];
    flags: IStateFlags;

    desktopDomainWithoutTld?: string;
    tlds?: Tld[];
    languages?: Lang[];
    mobile?: boolean;
    desktop?: boolean;
    specialDates?: StationDateSpecialValue[];
    events?: StationEventList[];
}

// Возвращает урлы страницы аэропорта в виде объектов для запросов.
// При необходимости можно переделать в функцию по получению списка урлов страницы станции
// для всех видов транспорта
export default function getDataForRequestsAirports({
    idsAirports,
    flags,

    desktopDomainWithoutTld = 'rasp.yandex',
    tlds = allTlds,
    languages = allLanguages,
    mobile = true,
    desktop = true,
    specialDates = allSpecialDates,
    events = allEvents,
}: IGetDataForRequestsAirports): IDataRequest[] {
    const languagesObject = languages.reduce<Partial<Record<Tld, true>>>(
        (result, lang) => {
            result[lang] = true;

            return result;
        },
        {},
    );

    return idsAirports.reduce<IDataRequest[]>((result, id) => {
        const params = cartesianProduct(tlds, specialDates, events);

        params.forEach(([tld, special, event]) => {
            getAvailableLanguages(tld)
                .filter(lang => languagesObject[lang])
                .forEach(language => {
                    const commonStationParams = {
                        id,
                        tld,
                        language,
                        special,
                        event,
                        flags,
                        type: StationType.plane,
                        subtypes: [StationSubtype.plane],
                        mainSubtype: StationSubtype.plane,
                        time: StationTime.all,
                    };

                    if (mobile) {
                        const host = getMobileHost(
                            desktopDomainWithoutTld,
                            tld,
                        );

                        const path = stationUrl({
                            ...commonStationParams,
                            isMobile: true,
                        });

                        result.push({host, path});
                    }

                    if (desktop) {
                        const host = getDesktopHost(
                            desktopDomainWithoutTld,
                            tld,
                        );

                        const path = stationUrl({
                            ...commonStationParams,
                            isMobile: false,
                        });

                        result.push({host, path});
                    }
                });
        });

        return result;
    }, []);
}
