/* eslint-disable no-console */
import fse from 'fs-extra';
import moment from 'moment';
import pathModule from 'path';
import yargs from 'yargs';

import IReport from '../../common/interfaces/tests/IReport';

import getDataForRequests from './dataForRequests/getDataForRequests';
import QueueFactoryOfPromise from '../../common/lib/QueueFactoryOfPromise';
import compareHttpCodes from './tasks/compareHttpCodes';
import Report from './report/Report';
import checkValidLinksOnPage from './tasks/checkValidLink';
import isRaspLink from '../../common/lib/url/isRaspLink';
import isNewStack from './tasks/isNewStack';

const argv = yargs
    .usage(
        'Usage: $0 --etalon=rasp.yandex --testing=experiments2.morda-front.rasp.common.yandex --maxConcurrent=3',
    )
    .help('h')
    .alias('h', 'help')
    .option('maxConcurrent', {
        type: 'number',
        describe: 'Максимальное количество одновременных запросов',
        demandOption: false,
        default: 1,
    })
    .option('etalon', {
        type: 'string',
        describe: 'Эталонное окружение',
        demandOption: true,
    })
    .option('testing', {
        type: 'string',
        describe: 'Тестируемое окружение',
        demandOption: true,
    }).argv;

function getEtalonHost(host: string): string {
    return host.replace('rasp.yandex', argv.etalon);
}

function getTestingHost(host: string): string {
    return host.replace('rasp.yandex', argv.testing);
}

function isStationPage(path: string): boolean {
    return /^\/station\//.test(path);
}

let countErrors = 0;

async function flushReport(
    report: IReport,
    pathToErrorFile: string,
    pathToInfoFile: string,
): Promise<void> {
    await fse.writeFile(pathToErrorFile, report.errorsToTxt(), {
        encoding: 'utf8',
        flag: 'a',
    });
    await fse.writeFile(pathToInfoFile, report.infotoTxt(), {
        encoding: 'utf8',
        flag: 'a',
    });

    countErrors += report.getCountErrors();

    report.clearErrors();
    report.clearInfo();
}

const uniqLinks = {};

function uniqueRaspLink(absoluteLink: string): boolean {
    if (uniqLinks[absoluteLink]) {
        return false;
    }

    uniqLinks[absoluteLink] = true;

    return isRaspLink(absoluteLink);
}

const errorsFile = pathModule.join(
    __dirname,
    '../../reports/integrationTestErrorsReport.txt',
);
const infoFile = pathModule.join(
    __dirname,
    '../../reports/integrationTestInfoReport.txt',
);

async function integrationTest(): Promise<void> {
    await fse.writeFile(infoFile, '', {encoding: 'utf8', flag: 'w'});
    await fse.writeFile(errorsFile, '', {encoding: 'utf8', flag: 'w'});
    const queue = new QueueFactoryOfPromise({
        maxConcurrent: 3,
        maxLength: 100000,
    });
    const report = new Report();

    queue.on('error', err => {
        report.pushError(err);
    });

    const dataForRequests = await getDataForRequests();
    const timeStart = moment();

    const printInterval = setInterval(() => {
        const countNotProcessed = queue.getLength();
        const countProcessed = queue.getProcessedLength();
        const totalCount = countNotProcessed + countProcessed;

        if (!countNotProcessed) {
            return null;
        }

        const elapsedMilliseconds: number =
            moment().valueOf() - timeStart.valueOf();
        const estimateMilliseconds: number = countProcessed
            ? Math.ceil(elapsedMilliseconds * (totalCount / countProcessed))
            : 0;
        const estimatedTime = moment(
            timeStart.valueOf() + estimateMilliseconds,
        );
        let info = `Всего: ${totalCount}. `;

        info += `Отстреляли: ${countProcessed} (${(
            countProcessed /
            (totalCount / 100)
        ).toFixed(1)}%). `;
        info += `${countErrors} ошибок. `;
        info += `Времени прошло: ${timeStart.fromNow(true)}. `;
        info += `Оценка выполнения: ${estimatedTime.format('HH:mm')} `;
        info += '        ';
        process.stdout.write(`\r${info}`);
    }, 1000);

    const flushReportInterval = setInterval(
        () => flushReport(report, errorsFile, infoFile),
        2000,
    );

    dataForRequests.forEach(({host, path}) => {
        const etalonHost = getEtalonHost(host);
        const testingHost = getTestingHost(host);

        if (isStationPage(path)) {
            queue.add(() =>
                compareHttpCodes(
                    [
                        {url: `https://${etalonHost}${path}`},
                        {url: `https://${testingHost}${path}`},
                    ],
                    report,
                ).then(dataPages => {
                    const dataStationPage = dataPages[dataPages.length - 1];

                    if (isNewStack(dataStationPage)) {
                        checkValidLinksOnPage(
                            dataStationPage,
                            report,
                            uniqueRaspLink,
                        ).forEach(task => queue.add(task));
                    }
                }),
            );
        }
    });

    await new Promise(resolve => {
        queue.on('end', resolve);
    });

    clearInterval(printInterval);
    clearInterval(flushReportInterval);

    await flushReport(report, errorsFile, infoFile);
}

integrationTest()
    .then(() => {
        console.log('Интеграционный тест завершен');
        process.exit(0);
    })
    .catch(error => {
        console.error(error);
        process.exit(1);
    });
