import path from 'path';
import webpack from 'webpack';
import {ReactLoadablePlugin} from 'react-loadable/webpack';
import {BundleAnalyzerPlugin} from 'webpack-bundle-analyzer';
import MiniCssExtractPlugin from 'mini-css-extract-plugin';

import IWebpackConfig from '../common/interfaces/IWebpackConfig';

import {
    getPathForBundle,
    getHostForBundle,
    getPathForReactLoadableJson,
} from './staticUtils';
import getDevtool from './getDevtool';
import getJsRules from './getJsRules';
import getCssRules from './getCssRules';
import getDefinePlugin from './getDefinePlugin';
import getStats from './getStats';
import getResolve from './getResolve';
import getSupportedBrowsers from './getSupportedBrowsers';
import getMinimizeOptimization from './getMinimizeOptimization';
import getImageRules from './getImageRules';
import getSvgBundlePlugin from './getSvgBundlePlugin';
import getPostCSSAssetsPlugin from './getPostCSSAssetsPlugin';
import getWriteFilePlugin from './getWriteFilePlugin';

const target = 'web';

export default function ({
    platform,
    language,
    mode,
    modernBrowser,
}: IWebpackConfig): Record<string, any> {
    const browsers = getSupportedBrowsers(platform, modernBrowser);
    const bundlePath = getPathForBundle({platform, language, modernBrowser});
    const bundleHost = getHostForBundle({platform, language, modernBrowser});
    const publicPath = `${bundleHost}/`;

    return {
        entry: {
            app: path.join(__dirname, '../client/app.js'),
        },
        output: {
            publicPath,
            path: bundlePath,
            filename: '[name].js',
            chunkFilename: '[name].chunk.js',
        },
        mode,
        target,
        watch: false,
        bail: true,
        profile: true,
        devtool: getDevtool(),
        stats: getStats(),
        resolve: getResolve(),
        optimization: {
            splitChunks: {
                chunks: 'all',
                cacheGroups: {
                    libs: {
                        test: /[\\/]node_modules[\\/]/,
                        name: 'libs',
                    },
                },
            },
            ...getMinimizeOptimization(mode),
        },
        module: {
            rules: [
                ...getJsRules(target, platform, modernBrowser),

                ...getCssRules(),

                ...getImageRules(),
            ],
        },
        plugins: [
            getSvgBundlePlugin('common/images'),

            getPostCSSAssetsPlugin(browsers, mode),

            new MiniCssExtractPlugin({
                filename: '[name].css',
                chunkFilename: '[name].chunk.css',
            }),

            new webpack.ContextReplacementPlugin(/moment[/\\]locale$/, /ru|uk/),

            getDefinePlugin({platform, language}),

            new ReactLoadablePlugin({
                filename: getPathForReactLoadableJson({
                    platform,
                    language,
                    modernBrowser,
                }),
            }),

            getWriteFilePlugin(),

            ...(process.env.BUNDLE_PROFILE
                ? [
                      new BundleAnalyzerPlugin({
                          analyzerMode: 'static',
                          defaultSizes: 'gzip',
                      }),
                  ]
                : []),
        ],
    };
}
