import path from 'path';
import MiniCssExtractPlugin from 'mini-css-extract-plugin';

import IWebpackConfig from '../common/interfaces/IWebpackConfig';
import Platform from '../common/interfaces/Platform';

import {
    getPathForInformerBundle,
    getHostForInformerBundle,
} from './staticUtils';
import getDevtool from './getDevtool';
import getJsRules from './getJsRules';
import getCssRules from './getCssRules';
import getDefinePlugin from './getDefinePlugin';
import getStats from './getStats';
import getResolve from './getResolve';
import getSupportedBrowsers from './getSupportedBrowsers';
import getMinimizeOptimization from './getMinimizeOptimization';
import getImageRules from './getImageRules';
import getSvgBundlePlugin from './getSvgBundlePlugin';
import getPostCSSAssetsPlugin from './getPostCSSAssetsPlugin';
import getWriteFilePlugin from './getWriteFilePlugin';

const target = 'web';

export default function ({
    language,
    mode,
}: IWebpackConfig): Record<string, any> {
    const browsers = getSupportedBrowsers();
    const bundlePath = getPathForInformerBundle({language});
    const bundleHost = getHostForInformerBundle({language});
    const publicPath = `${bundleHost}/`;

    return {
        entry: {
            app: path.join(
                __dirname,
                '../common/components/RootInformers/RootInformers.tsx',
            ),
        },
        output: {
            publicPath,
            path: bundlePath,
            filename: '[name].js',
        },
        mode,
        target,
        watch: false,
        bail: true,
        profile: true,
        devtool: getDevtool(),
        stats: getStats(),
        resolve: getResolve(),
        optimization: getMinimizeOptimization(mode),
        module: {
            rules: [
                // Максимально упрощаем сборку клиентского js, потому что он нам не нужен, тк информеры статические
                ...getJsRules(target, Platform.desktop, true),

                ...getCssRules(),

                ...getImageRules(),
            ],
        },
        plugins: [
            getSvgBundlePlugin('common/informerImages'),

            getPostCSSAssetsPlugin(browsers, mode),

            new MiniCssExtractPlugin({
                filename: '[name].css',
            }),

            getDefinePlugin({platform: Platform.desktop, language}),

            getWriteFilePlugin(),
        ],
    };
}
