import fs from 'fs-extra';
import environment from '@yandex-int/yandex-environment';

import IWebpackConfig from '../common/interfaces/IWebpackConfig';
import Platform from '../common/interfaces/Platform';
import Lang from '../common/interfaces/Lang';
import WebpackMode from '../common/interfaces/WebpackMode';
import Environment from '../common/interfaces/Environment';

import {generateMSApplicationConfig} from '../server/helpers/favicons';
import {
    getPathForBundle,
    getPathForFavicons,
    getPathForInformerBundle,
} from './staticUtils';
import clientConfig from './client.config';
import serverConfig from './server.config';
import serverInformerConfig from './serverInformer.config';
import clientInformerConfig from './clientInformer.config';
import getValueFromEnum from '../common/lib/enum/getValueFromEnum';

export default function (params?: IWebpackConfig): Record<string, any>[] {
    const platform =
        params?.platform ||
        (process.env.PLATFORM &&
            getValueFromEnum(process.env.PLATFORM, Platform)) ||
        Platform.desktop;
    const mode =
        params?.mode || environment === Environment.development
            ? WebpackMode.development
            : WebpackMode.production;
    const modernBrowser = Boolean(
        params?.modernBrowser ?? process.env.MODERN_BROWSER === '1',
    );
    const language =
        params?.language ||
        (process.env.BUNDLE_LANGUAGE &&
            getValueFromEnum(process.env.BUNDLE_LANGUAGE, Lang)) ||
        Lang.ru;

    const bundlePath = getPathForBundle({platform, language, modernBrowser});
    const informerBundlePath = getPathForInformerBundle({language});

    // убеждаемся в наличии папки для сборки и чистим ее содержимое
    fs.emptyDirSync(bundlePath);
    fs.emptyDirSync(informerBundlePath);

    // создаем файл с настройками отображения сайта для винды
    fs.writeFileSync(
        `${getPathForFavicons()}/browserConfig.xml`,
        generateMSApplicationConfig(),
    );

    const configOptions: IWebpackConfig = {
        platform,
        language,
        mode,
        modernBrowser,
    };

    return [
        clientConfig(configOptions),
        serverConfig(configOptions),
        ...(platform === Platform.desktop && !modernBrowser
            ? [
                  serverInformerConfig(configOptions),
                  clientInformerConfig(configOptions),
              ]
            : []),
    ];
}
