import path from 'path';
import fs from 'fs-extra';
import environment from '@yandex-int/yandex-environment';
import {PluginItem, TransformOptions} from '@babel/core';

import Platform from '../common/interfaces/Platform';
import Environment from '../common/interfaces/Environment';

import getSupportedBrowsers from './getSupportedBrowsers';
import {patchOptionsPresetAndPlugins} from './babelUtils';

const babelrcPath = path.join(__dirname, '../.babelrc');
const babelrc: TransformOptions = fs.readJsonSync(babelrcPath);
const isDev = environment === Environment.development;

function getPresets(
    target: string,

    platform?: Platform,
    modernBrowser = false,
): PluginItem[] {
    const presets = [...(babelrc.presets || [])];

    if (target === 'node') {
        return presets;
    }

    const browsers = getSupportedBrowsers(platform, modernBrowser);

    return patchOptionsPresetAndPlugins(presets, '@babel/preset-env', {
        targets: {
            browsers,
        },
        modules: false,
        useBuiltIns: 'usage',
        corejs: 2,
    });
}

function getPlugins(target: string): PluginItem[] {
    let plugins = [...(babelrc.plugins || [])];

    if (target === 'web') {
        plugins = plugins
            // добавлем динамические импорты для клиентской части
            .filter(plugin => plugin !== 'babel-plugin-dynamic-import-node')
            .concat('@babel/plugin-syntax-dynamic-import');
    }

    if (isDev) {
        plugins = plugins.filter(
            plugin =>
                plugin !== 'babel-plugin-transform-react-remove-prop-types',
        );
    }

    return plugins;
}

export default function (
    target: string,

    platform?: Platform,
    modernBrowser = false,
): any[] {
    const presets = getPresets(target, platform, modernBrowser);
    const plugins = getPlugins(target);

    return [
        {
            test: /\.([jt]sx?|mjs)$/,
            exclude: [
                /node_modules/,
                path.join(__dirname, '../browserTests'),
                path.join(__dirname, '../static'),
            ],
            use: [
                {
                    loader: 'babel-loader',
                    options: {
                        cacheDirectory: true,
                        babelrc: false,
                        presets,
                        plugins,
                    },
                },
            ],
        },
    ];
}
