import {Agent} from 'https';
import S3Client from 'aws-sdk/clients/s3';

import {pinnedMessagesKey} from '../configs/constants';

import IS3Config from '../types/IApiConfig/types/IS3Config';
import IChatIdAndMessageId from '../types/IChatIdAndMessageId';

import logger from '../utilities/logger';

export default class S3Storage {
    private client: S3Client;
    private bucketName: string;

    constructor({chatBucket, client}: IS3Config) {
        this.client = new S3Client({
            endpoint: client.baseUrl,
            accessKeyId: client.accessKey,
            secretAccessKey: client.secretAccessKey,
            httpOptions: {
                agent: new Agent({
                    rejectUnauthorized: false,
                }),
            },
        });

        this.bucketName = chatBucket;
    }

    async saveChatIdByAccount(account: string, chatId: number): Promise<void> {
        try {
            await this.client
                .putObject({
                    Bucket: this.bucketName,
                    Key: account.toLowerCase(),
                    Body: `${chatId}`,
                })
                .promise();

            logger.log(`Save chat id for ${account} success`);
        } catch (e) {
            if (e instanceof Error) {
                logger.logError(`Save chat id for ${account} fail`, e);
            }

            throw e;
        }
    }

    async getChatIdByAccount(account: string): Promise<string | undefined> {
        try {
            const data = await this.client
                .getObject({
                    Bucket: this.bucketName,
                    Key: account.toLowerCase(),
                })
                .promise();

            return data.Body?.toString();
        } catch (e) {
            if (e instanceof Error) {
                logger.logError('Failed to get chat id by account', e);
            }
        }
    }

    async setPinnedMessages(
        pinnedMessages: IChatIdAndMessageId[],
    ): Promise<boolean> {
        try {
            await this.client
                .putObject({
                    Bucket: this.bucketName,
                    Key: pinnedMessagesKey,
                    Body: JSON.stringify(pinnedMessages),
                })
                .promise();

            return true;
        } catch (e) {
            if (e instanceof Error) {
                logger.logError('Failed to save pinned chats in S3', e);
            }

            return false;
        }
    }

    async getPinnedMessages(): Promise<IChatIdAndMessageId[]> {
        try {
            const data = await this.client
                .getObject({Bucket: this.bucketName, Key: pinnedMessagesKey})
                .promise();
            const str = data.Body?.toString();

            if (!str) {
                return [];
            }

            return JSON.parse(str);
        } catch (e) {
            if (e instanceof Error) {
                logger.logError('Failed to get pinned chats from S3', e);
            }

            return [];
        }
    }
}
